<?php
/**
 * Kkart Order Functions
 *
 * Functions for order specific things.
 *
 * @package Kkart\Functions
 * @version 3.4.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Standard way of retrieving orders based on certain parameters.
 *
 * This function should be used for order retrieval so that when we move to
 * custom tables, functions still work.
 *
 * Args and usage: https://github.com/kkart/kkart/wiki/kkart_get_orders-and-KKART_Order_Query
 *
 * @since  2.6.0
 * @param  array $args Array of args (above).
 * @return KKART_Order[]|stdClass Number of pages and an array of order objects if
 *                             paginate is true, or just an array of values.
 */
function kkart_get_orders( $args ) {
	$map_legacy = array(
		'numberposts'    => 'limit',
		'post_type'      => 'type',
		'post_status'    => 'status',
		'post_parent'    => 'parent',
		'author'         => 'customer',
		'email'          => 'billing_email',
		'posts_per_page' => 'limit',
		'paged'          => 'page',
	);

	foreach ( $map_legacy as $from => $to ) {
		if ( isset( $args[ $from ] ) ) {
			$args[ $to ] = $args[ $from ];
		}
	}

	// Map legacy date args to modern date args.
	$date_before = false;
	$date_after  = false;

	if ( ! empty( $args['date_before'] ) ) {
		$datetime    = kkart_string_to_datetime( $args['date_before'] );
		$date_before = strpos( $args['date_before'], ':' ) ? $datetime->getOffsetTimestamp() : $datetime->date( 'Y-m-d' );
	}
	if ( ! empty( $args['date_after'] ) ) {
		$datetime   = kkart_string_to_datetime( $args['date_after'] );
		$date_after = strpos( $args['date_after'], ':' ) ? $datetime->getOffsetTimestamp() : $datetime->date( 'Y-m-d' );
	}

	if ( $date_before && $date_after ) {
		$args['date_created'] = $date_after . '...' . $date_before;
	} elseif ( $date_before ) {
		$args['date_created'] = '<' . $date_before;
	} elseif ( $date_after ) {
		$args['date_created'] = '>' . $date_after;
	}

	$query = new KKART_Order_Query( $args );
	return $query->get_orders();
}

/**
 * Main function for returning orders, uses the KKART_Order_Factory class.
 *
 * @since  2.2
 *
 * @param  mixed $the_order Post object or post ID of the order.
 *
 * @return bool|KKART_Order|KKART_Order_Refund
 */
function kkart_get_order( $the_order = false ) {
	if ( ! did_action( 'kkart_after_register_post_type' ) ) {
		kkart_doing_it_wrong( __FUNCTION__, 'kkart_get_order should not be called before post types are registered (kkart_after_register_post_type action)', '2.5' );
		return false;
	}
	return KKART()->order_factory->get_order( $the_order );
}

/**
 * Get all order statuses.
 *
 * @since 2.2
 * @used-by KKART_Order::set_status
 * @return array
 */
function kkart_get_order_statuses() {
	$order_statuses = array(
		'kkart-pending'    => _x( 'Pending payment', 'Order status', 'kkart' ),
		'kkart-processing' => _x( 'Processing', 'Order status', 'kkart' ),
		'kkart-on-hold'    => _x( 'On hold', 'Order status', 'kkart' ),
		'kkart-completed'  => _x( 'Completed', 'Order status', 'kkart' ),
		'kkart-cancelled'  => _x( 'Cancelled', 'Order status', 'kkart' ),
		'kkart-refunded'   => _x( 'Refunded', 'Order status', 'kkart' ),
		'kkart-failed'     => _x( 'Failed', 'Order status', 'kkart' ),
	);
	return apply_filters( 'kkart_order_statuses', $order_statuses );
}

/**
 * See if a string is an order status.
 *
 * @param  string $maybe_status Status, including any kkart- prefix.
 * @return bool
 */
function kkart_is_order_status( $maybe_status ) {
	$order_statuses = kkart_get_order_statuses();
	return isset( $order_statuses[ $maybe_status ] );
}

/**
 * Get list of statuses which are consider 'paid'.
 *
 * @since  3.0.0
 * @return array
 */
function kkart_get_is_paid_statuses() {
	return apply_filters( 'kkart_order_is_paid_statuses', array( 'processing', 'completed' ) );
}

/**
 * Get list of statuses which are consider 'pending payment'.
 *
 * @since  3.6.0
 * @return array
 */
function kkart_get_is_pending_statuses() {
	return apply_filters( 'kkart_order_is_pending_statuses', array( 'pending' ) );
}

/**
 * Get the nice name for an order status.
 *
 * @since  2.2
 * @param  string $status Status.
 * @return string
 */
function kkart_get_order_status_name( $status ) {
	$statuses = kkart_get_order_statuses();
	$status   = 'kkart-' === substr( $status, 0, 6 ) ? substr( $status, 6 ) : $status;
	$status   = isset( $statuses[ 'kkart-' . $status ] ) ? $statuses[ 'kkart-' . $status ] : $status;
	return $status;
}

/**
 * Generate an order key with prefix.
 *
 * @since 3.5.4
 * @param string $key Order key without a prefix. By default generates a 13 digit secret.
 * @return string The order key.
 */
function kkart_generate_order_key( $key = '' ) {
	if ( '' === $key ) {
		$key = wp_generate_password( 13, false );
	}

	return 'kkart_' . apply_filters( 'kkart_generate_order_key', 'order_' . $key );
}

/**
 * Finds an Order ID based on an order key.
 *
 * @param string $order_key An order key has generated by.
 * @return int The ID of an order, or 0 if the order could not be found.
 */
function kkart_get_order_id_by_order_key( $order_key ) {
	$data_store = KKART_Data_Store::load( 'order' );
	return $data_store->get_order_id_by_order_key( $order_key );
}

/**
 * Get all registered order types.
 *
 * @since  2.2
 * @param  string $for Optionally define what you are getting order types for so
 *                     only relevant types are returned.
 *                     e.g. for 'order-meta-boxes', 'order-count'.
 * @return array
 */
function kkart_get_order_types( $for = '' ) {
	global $kkart_order_types;

	if ( ! is_array( $kkart_order_types ) ) {
		$kkart_order_types = array();
	}

	$order_types = array();

	switch ( $for ) {
		case 'order-count':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( ! $args['exclude_from_order_count'] ) {
					$order_types[] = $type;
				}
			}
			break;
		case 'order-meta-boxes':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( $args['add_order_meta_boxes'] ) {
					$order_types[] = $type;
				}
			}
			break;
		case 'view-orders':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( ! $args['exclude_from_order_views'] ) {
					$order_types[] = $type;
				}
			}
			break;
		case 'reports':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( ! $args['exclude_from_order_reports'] ) {
					$order_types[] = $type;
				}
			}
			break;
		case 'sales-reports':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( ! $args['exclude_from_order_sales_reports'] ) {
					$order_types[] = $type;
				}
			}
			break;
		case 'order-webhooks':
			foreach ( $kkart_order_types as $type => $args ) {
				if ( ! $args['exclude_from_order_webhooks'] ) {
					$order_types[] = $type;
				}
			}
			break;
		default:
			$order_types = array_keys( $kkart_order_types );
			break;
	}

	return apply_filters( 'kkart_order_types', $order_types, $for );
}

/**
 * Get an order type by post type name.
 *
 * @param  string $type Post type name.
 * @return bool|array Details about the order type.
 */
function kkart_get_order_type( $type ) {
	global $kkart_order_types;

	if ( isset( $kkart_order_types[ $type ] ) ) {
		return $kkart_order_types[ $type ];
	}

	return false;
}

/**
 * Register order type. Do not use before init.
 *
 * Wrapper for register post type, as well as a method of telling KKART which.
 * post types are types of orders, and having them treated as such.
 *
 * $args are passed to register_post_type, but there are a few specific to this function:
 *      - exclude_from_orders_screen (bool) Whether or not this order type also get shown in the main.
 *      orders screen.
 *      - add_order_meta_boxes (bool) Whether or not the order type gets shop_order meta boxes.
 *      - exclude_from_order_count (bool) Whether or not this order type is excluded from counts.
 *      - exclude_from_order_views (bool) Whether or not this order type is visible by customers when.
 *      viewing orders e.g. on the my account page.
 *      - exclude_from_order_reports (bool) Whether or not to exclude this type from core reports.
 *      - exclude_from_order_sales_reports (bool) Whether or not to exclude this type from core sales reports.
 *
 * @since  2.2
 * @see    register_post_type for $args used in that function
 * @param  string $type Post type. (max. 20 characters, can not contain capital letters or spaces).
 * @param  array  $args An array of arguments.
 * @return bool Success or failure
 */
function kkart_register_order_type( $type, $args = array() ) {
	if ( post_type_exists( $type ) ) {
		return false;
	}

	global $kkart_order_types;

	if ( ! is_array( $kkart_order_types ) ) {
		$kkart_order_types = array();
	}

	// Register as a post type.
	if ( is_wp_error( register_post_type( $type, $args ) ) ) {
		return false;
	}

	// Register for KKART usage.
	$order_type_args = array(
		'exclude_from_orders_screen'       => false,
		'add_order_meta_boxes'             => true,
		'exclude_from_order_count'         => false,
		'exclude_from_order_views'         => false,
		'exclude_from_order_webhooks'      => false,
		'exclude_from_order_reports'       => false,
		'exclude_from_order_sales_reports' => false,
		'class_name'                       => 'KKART_Order',
	);

	$args                    = array_intersect_key( $args, $order_type_args );
	$args                    = wp_parse_args( $args, $order_type_args );
	$kkart_order_types[ $type ] = $args;

	return true;
}

/**
 * Return the count of processing orders.
 *
 * @return int
 */
function kkart_processing_order_count() {
	return kkart_orders_count( 'processing' );
}

/**
 * Return the orders count of a specific order status.
 *
 * @param string $status Status.
 * @return int
 */
function kkart_orders_count( $status ) {
	$count          = 0;
	$status         = 'kkart-' . $status;
	$order_statuses = array_keys( kkart_get_order_statuses() );

	if ( ! in_array( $status, $order_statuses, true ) ) {
		return 0;
	}

	$cache_key    = KKART_Cache_Helper::get_cache_prefix( 'orders' ) . $status;
	$cached_count = wp_cache_get( $cache_key, 'counts' );

	if ( false !== $cached_count ) {
		return $cached_count;
	}

	foreach ( kkart_get_order_types( 'order-count' ) as $type ) {
		$data_store = KKART_Data_Store::load( 'shop_order' === $type ? 'order' : $type );
		if ( $data_store ) {
			$count += $data_store->get_order_count( $status );
		}
	}

	wp_cache_set( $cache_key, $count, 'counts' );

	return $count;
}

/**
 * Grant downloadable product access to the file identified by $download_id.
 *
 * @param  string         $download_id File identifier.
 * @param  int|KKART_Product $product     Product instance or ID.
 * @param  KKART_Order       $order       Order data.
 * @param  int            $qty         Quantity purchased.
 * @return int|bool insert id or false on failure.
 */
function kkart_downloadable_file_permission( $download_id, $product, $order, $qty = 1 ) {
	if ( is_numeric( $product ) ) {
		$product = kkart_get_product( $product );
	}
	$download = new KKART_Customer_Download();
	$download->set_download_id( $download_id );
	$download->set_product_id( $product->get_id() );
	$download->set_user_id( $order->get_customer_id() );
	$download->set_order_id( $order->get_id() );
	$download->set_user_email( $order->get_billing_email() );
	$download->set_order_key( $order->get_order_key() );
	$download->set_downloads_remaining( 0 > $product->get_download_limit() ? '' : $product->get_download_limit() * $qty );
	$download->set_access_granted( time() );
	$download->set_download_count( 0 );

	$expiry = $product->get_download_expiry();

	if ( $expiry > 0 ) {
		$from_date = $order->get_date_completed() ? $order->get_date_completed()->format( 'Y-m-d' ) : current_time( 'mysql', true );
		$download->set_access_expires( strtotime( $from_date . ' + ' . $expiry . ' DAY' ) );
	}

	$download = apply_filters( 'kkart_downloadable_file_permission', $download, $product, $order, $qty );

	return $download->save();
}

/**
 * Order Status completed - give downloadable product access to customer.
 *
 * @param int  $order_id Order ID.
 * @param bool $force    Force downloadable permissions.
 */
function kkart_downloadable_product_permissions( $order_id, $force = false ) {
	$order = kkart_get_order( $order_id );

	if ( ! $order || ( $order->get_data_store()->get_download_permissions_granted( $order ) && ! $force ) ) {
		return;
	}

	if ( $order->has_status( 'processing' ) && 'no' === get_option( 'kkart_downloads_grant_access_after_payment' ) ) {
		return;
	}

	if ( count( $order->get_items() ) > 0 ) {
		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();

			if ( $product && $product->exists() && $product->is_downloadable() ) {
				$downloads = $product->get_downloads();

				foreach ( array_keys( $downloads ) as $download_id ) {
					kkart_downloadable_file_permission( $download_id, $product, $order, $item->get_quantity() );
				}
			}
		}
	}

	$order->get_data_store()->set_download_permissions_granted( $order, true );
	do_action( 'kkart_grant_product_download_permissions', $order_id );
}
add_action( 'kkart_order_status_completed', 'kkart_downloadable_product_permissions' );
add_action( 'kkart_order_status_processing', 'kkart_downloadable_product_permissions' );

/**
 * Clear all transients cache for order data.
 *
 * @param int|KKART_Order $order Order instance or ID.
 */
function kkart_delete_shop_order_transients( $order = 0 ) {
	if ( is_numeric( $order ) ) {
		$order = kkart_get_order( $order );
	}
	$reports             = KKART_Admin_Reports::get_reports();
	$transients_to_clear = array(
		'kkart_admin_report',
	);

	foreach ( $reports as $report_group ) {
		foreach ( $report_group['reports'] as $report_key => $report ) {
			$transients_to_clear[] = 'kkart_report_' . $report_key;
		}
	}

	foreach ( $transients_to_clear as $transient ) {
		delete_transient( $transient );
	}

	// Clear money spent for user associated with order.
	if ( is_a( $order, 'KKART_Order' ) ) {
		$order_id = $order->get_id();
		delete_user_meta( $order->get_customer_id(), '_money_spent' );
		delete_user_meta( $order->get_customer_id(), '_order_count' );
	} else {
		$order_id = 0;
	}

	// Increments the transient version to invalidate cache.
	KKART_Cache_Helper::get_transient_version( 'orders', true );

	// Do the same for regular cache.
	KKART_Cache_Helper::invalidate_cache_group( 'orders' );

	do_action( 'kkart_delete_shop_order_transients', $order_id );
}

/**
 * See if we only ship to billing addresses.
 *
 * @return bool
 */
function kkart_ship_to_billing_address_only() {
	return 'billing_only' === get_option( 'kkart_ship_to_destination' );
}

/**
 * Create a new order refund programmatically.
 *
 * Returns a new refund object on success which can then be used to add additional data.
 *
 * @since 2.2
 * @throws Exception Throws exceptions when fail to create, but returns WP_Error instead.
 * @param array $args New refund arguments.
 * @return KKART_Order_Refund|WP_Error
 */
function kkart_create_refund( $args = array() ) {
	$default_args = array(
		'amount'         => 0,
		'reason'         => null,
		'order_id'       => 0,
		'refund_id'      => 0,
		'line_items'     => array(),
		'refund_payment' => false,
		'restock_items'  => false,
	);

	try {
		$args  = wp_parse_args( $args, $default_args );
		$order = kkart_get_order( $args['order_id'] );

		if ( ! $order ) {
			throw new Exception( __( 'Invalid order ID.', 'kkart' ) );
		}

		$remaining_refund_amount = $order->get_remaining_refund_amount();
		$remaining_refund_items  = $order->get_remaining_refund_items();
		$refund_item_count       = 0;
		$refund                  = new KKART_Order_Refund( $args['refund_id'] );

		if ( 0 > $args['amount'] || $args['amount'] > $remaining_refund_amount ) {
			throw new Exception( __( 'Invalid refund amount.', 'kkart' ) );
		}

		$refund->set_currency( $order->get_currency() );
		$refund->set_amount( $args['amount'] );
		$refund->set_parent_id( absint( $args['order_id'] ) );
		$refund->set_refunded_by( get_current_user_id() ? get_current_user_id() : 1 );
		$refund->set_prices_include_tax( $order->get_prices_include_tax() );

		if ( ! is_null( $args['reason'] ) ) {
			$refund->set_reason( $args['reason'] );
		}

		// Negative line items.
		if ( count( $args['line_items'] ) > 0 ) {
			$items = $order->get_items( array( 'line_item', 'fee', 'shipping' ) );

			foreach ( $items as $item_id => $item ) {
				if ( ! isset( $args['line_items'][ $item_id ] ) ) {
					continue;
				}

				$qty          = isset( $args['line_items'][ $item_id ]['qty'] ) ? $args['line_items'][ $item_id ]['qty'] : 0;
				$refund_total = $args['line_items'][ $item_id ]['refund_total'];
				$refund_tax   = isset( $args['line_items'][ $item_id ]['refund_tax'] ) ? array_filter( (array) $args['line_items'][ $item_id ]['refund_tax'] ) : array();

				if ( empty( $qty ) && empty( $refund_total ) && empty( $args['line_items'][ $item_id ]['refund_tax'] ) ) {
					continue;
				}

				$class         = get_class( $item );
				$refunded_item = new $class( $item );
				$refunded_item->set_id( 0 );
				$refunded_item->add_meta_data( '_refunded_item_id', $item_id, true );
				$refunded_item->set_total( kkart_format_refund_total( $refund_total ) );
				$refunded_item->set_taxes(
					array(
						'total'    => array_map( 'kkart_format_refund_total', $refund_tax ),
						'subtotal' => array_map( 'kkart_format_refund_total', $refund_tax ),
					)
				);

				if ( is_callable( array( $refunded_item, 'set_subtotal' ) ) ) {
					$refunded_item->set_subtotal( kkart_format_refund_total( $refund_total ) );
				}

				if ( is_callable( array( $refunded_item, 'set_quantity' ) ) ) {
					$refunded_item->set_quantity( $qty * -1 );
				}

				$refund->add_item( $refunded_item );
				$refund_item_count += $qty;
			}
		}

		$refund->update_taxes();
		$refund->calculate_totals( false );
		$refund->set_total( $args['amount'] * -1 );

		// this should remain after update_taxes(), as this will save the order, and write the current date to the db
		// so we must wait until the order is persisted to set the date.
		if ( isset( $args['date_created'] ) ) {
			$refund->set_date_created( $args['date_created'] );
		}

		/**
		 * Action hook to adjust refund before save.
		 *
		 * @since 3.0.0
		 */
		do_action( 'kkart_create_refund', $refund, $args );

		if ( $refund->save() ) {
			if ( $args['refund_payment'] ) {
				$result = kkart_refund_payment( $order, $refund->get_amount(), $refund->get_reason() );

				if ( is_wp_error( $result ) ) {
					$refund->delete();
					return $result;
				}

				$refund->set_refunded_payment( true );
				$refund->save();
			}

			if ( $args['restock_items'] ) {
				kkart_restock_refunded_items( $order, $args['line_items'] );
			}

			// Trigger notification emails.
			if ( ( $remaining_refund_amount - $args['amount'] ) > 0 || ( $order->has_free_item() && ( $remaining_refund_items - $refund_item_count ) > 0 ) ) {
				do_action( 'kkart_order_partially_refunded', $order->get_id(), $refund->get_id() );
			} else {
				do_action( 'kkart_order_fully_refunded', $order->get_id(), $refund->get_id() );

				$parent_status = apply_filters( 'kkart_order_fully_refunded_status', 'refunded', $order->get_id(), $refund->get_id() );

				if ( $parent_status ) {
					$order->update_status( $parent_status );
				}
			}
		}

		do_action( 'kkart_refund_created', $refund->get_id(), $args );
		do_action( 'kkart_order_refunded', $order->get_id(), $refund->get_id() );

	} catch ( Exception $e ) {
		if ( isset( $refund ) && is_a( $refund, 'KKART_Order_Refund' ) ) {
			wp_delete_post( $refund->get_id(), true );
		}
		return new WP_Error( 'error', $e->getMessage() );
	}

	return $refund;
}

/**
 * Try to refund the payment for an order via the gateway.
 *
 * @since 3.0.0
 * @throws Exception Throws exceptions when fail to refund, but returns WP_Error instead.
 * @param KKART_Order $order  Order instance.
 * @param string   $amount Amount to refund.
 * @param string   $reason Refund reason.
 * @return bool|WP_Error
 */
function kkart_refund_payment( $order, $amount, $reason = '' ) {
	try {
		if ( ! is_a( $order, 'KKART_Order' ) ) {
			throw new Exception( __( 'Invalid order.', 'kkart' ) );
		}

		$gateway_controller = KKART_Payment_Gateways::instance();
		$all_gateways       = $gateway_controller->payment_gateways();
		$payment_method     = $order->get_payment_method();
		$gateway            = isset( $all_gateways[ $payment_method ] ) ? $all_gateways[ $payment_method ] : false;

		if ( ! $gateway ) {
			throw new Exception( __( 'The payment gateway for this order does not exist.', 'kkart' ) );
		}

		if ( ! $gateway->supports( 'refunds' ) ) {
			throw new Exception( __( 'The payment gateway for this order does not support automatic refunds.', 'kkart' ) );
		}

		$result = $gateway->process_refund( $order->get_id(), $amount, $reason );

		if ( ! $result ) {
			throw new Exception( __( 'An error occurred while attempting to create the refund using the payment gateway API.', 'kkart' ) );
		}

		if ( is_wp_error( $result ) ) {
			throw new Exception( $result->get_error_message() );
		}

		return true;

	} catch ( Exception $e ) {
		return new WP_Error( 'error', $e->getMessage() );
	}
}

/**
 * Restock items during refund.
 *
 * @since 3.0.0
 * @param KKART_Order $order               Order instance.
 * @param array    $refunded_line_items Refunded items list.
 */
function kkart_restock_refunded_items( $order, $refunded_line_items ) {
	if ( ! apply_filters( 'kkart_can_restock_refunded_items', true, $order, $refunded_line_items ) ) {
		return;
	}

	$line_items = $order->get_items();

	foreach ( $line_items as $item_id => $item ) {
		if ( ! isset( $refunded_line_items[ $item_id ], $refunded_line_items[ $item_id ]['qty'] ) ) {
			continue;
		}
		$product            = $item->get_product();
		$item_stock_reduced = $item->get_meta( '_reduced_stock', true );
		$qty_to_refund      = $refunded_line_items[ $item_id ]['qty'];

		if ( ! $item_stock_reduced || ! $qty_to_refund || ! $product || ! $product->managing_stock() ) {
			continue;
		}

		$old_stock = $product->get_stock_quantity();
		$new_stock = kkart_update_product_stock( $product, $qty_to_refund, 'increase' );

		// Update _reduced_stock meta to track changes.
		$item_stock_reduced = $item_stock_reduced - $qty_to_refund;

		if ( 0 < $item_stock_reduced ) {
			$item->update_meta_data( '_reduced_stock', $item_stock_reduced );
		} else {
			$item->delete_meta_data( '_reduced_stock' );
		}

		/* translators: 1: product ID 2: old stock level 3: new stock level */
		$order->add_order_note( sprintf( __( 'Item #%1$s stock increased from %2$s to %3$s.', 'kkart' ), $product->get_id(), $old_stock, $new_stock ) );

		$item->save();

		do_action( 'kkart_restock_refunded_item', $product->get_id(), $old_stock, $new_stock, $order, $product );
	}
}

/**
 * Get tax class by tax id.
 *
 * @since 2.2
 * @param int $tax_id Tax ID.
 * @return string
 */
function kkart_get_tax_class_by_tax_id( $tax_id ) {
	global $wpdb;
	return $wpdb->get_var( $wpdb->prepare( "SELECT tax_rate_class FROM {$wpdb->prefix}kkart_tax_rates WHERE tax_rate_id = %d", $tax_id ) );
}

/**
 * Get payment gateway class by order data.
 *
 * @since 2.2
 * @param int|KKART_Order $order Order instance.
 * @return KKART_Payment_Gateway|bool
 */
function kkart_get_payment_gateway_by_order( $order ) {
	if ( KKART()->payment_gateways() ) {
		$payment_gateways = KKART()->payment_gateways()->payment_gateways();
	} else {
		$payment_gateways = array();
	}

	if ( ! is_object( $order ) ) {
		$order_id = absint( $order );
		$order    = kkart_get_order( $order_id );
	}

	return is_a( $order, 'KKART_Order' ) && isset( $payment_gateways[ $order->get_payment_method() ] ) ? $payment_gateways[ $order->get_payment_method() ] : false;
}

/**
 * When refunding an order, create a refund line item if the partial refunds do not match order total.
 *
 * This is manual; no gateway refund will be performed.
 *
 * @since 2.4
 * @param int $order_id Order ID.
 */
function kkart_order_fully_refunded( $order_id ) {
	$order      = kkart_get_order( $order_id );
	$max_refund = kkart_format_decimal( $order->get_total() - $order->get_total_refunded() );

	if ( ! $max_refund ) {
		return;
	}

	// Create the refund object.
	kkart_switch_to_site_locale();
	kkart_create_refund(
		array(
			'amount'     => $max_refund,
			'reason'     => __( 'Order fully refunded.', 'kkart' ),
			'order_id'   => $order_id,
			'line_items' => array(),
		)
	);
	kkart_restore_locale();

	$order->add_order_note( __( 'Order status set to refunded. To return funds to the customer you will need to issue a refund through your payment gateway.', 'kkart' ) );
}
add_action( 'kkart_order_status_refunded', 'kkart_order_fully_refunded' );

/**
 * Search orders.
 *
 * @since  2.6.0
 * @param  string $term Term to search.
 * @return array List of orders ID.
 */
function kkart_order_search( $term ) {
	$data_store = KKART_Data_Store::load( 'order' );
	return $data_store->search_orders( str_replace( 'Order #', '', kkart_clean( $term ) ) );
}

/**
 * Update total sales amount for each product within a paid order.
 *
 * @since 3.0.0
 * @param int $order_id Order ID.
 */
function kkart_update_total_sales_counts( $order_id ) {
	$order = kkart_get_order( $order_id );

	if ( ! $order || $order->get_data_store()->get_recorded_sales( $order ) ) {
		return;
	}

	if ( count( $order->get_items() ) > 0 ) {
		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_product_id();

			if ( $product_id ) {
				$data_store = KKART_Data_Store::load( 'product' );
				$data_store->update_product_sales( $product_id, absint( $item->get_quantity() ), 'increase' );
			}
		}
	}

	$order->get_data_store()->set_recorded_sales( $order, true );

	/**
	 * Called when sales for an order are recorded
	 *
	 * @param int $order_id order id
	 */
	do_action( 'kkart_recorded_sales', $order_id );
}
add_action( 'kkart_order_status_completed', 'kkart_update_total_sales_counts' );
add_action( 'kkart_order_status_processing', 'kkart_update_total_sales_counts' );
add_action( 'kkart_order_status_on-hold', 'kkart_update_total_sales_counts' );

/**
 * Update used coupon amount for each coupon within an order.
 *
 * @since 3.0.0
 * @param int $order_id Order ID.
 */
function kkart_update_coupon_usage_counts( $order_id ) {
	$order = kkart_get_order( $order_id );

	if ( ! $order ) {
		return;
	}

	$has_recorded = $order->get_data_store()->get_recorded_coupon_usage_counts( $order );

	if ( $order->has_status( 'cancelled' ) && $has_recorded ) {
		$action = 'reduce';
		$order->get_data_store()->set_recorded_coupon_usage_counts( $order, false );
	} elseif ( ! $order->has_status( 'cancelled' ) && ! $has_recorded ) {
		$action = 'increase';
		$order->get_data_store()->set_recorded_coupon_usage_counts( $order, true );
	} elseif ( $order->has_status( 'cancelled' ) ) {
		$order->get_data_store()->release_held_coupons( $order, true );
		return;
	} else {
		return;
	}

	if ( count( $order->get_coupon_codes() ) > 0 ) {
		foreach ( $order->get_coupon_codes() as $code ) {
			if ( ! $code ) {
				continue;
			}

			$coupon  = new KKART_Coupon( $code );
			$used_by = $order->get_user_id();

			if ( ! $used_by ) {
				$used_by = $order->get_billing_email();
			}

			switch ( $action ) {
				case 'reduce':
					$coupon->decrease_usage_count( $used_by );
					break;
				case 'increase':
					$coupon->increase_usage_count( $used_by, $order );
					break;
			}
		}
		$order->get_data_store()->release_held_coupons( $order, true );
	}
}
add_action( 'kkart_order_status_pending', 'kkart_update_coupon_usage_counts' );
add_action( 'kkart_order_status_completed', 'kkart_update_coupon_usage_counts' );
add_action( 'kkart_order_status_processing', 'kkart_update_coupon_usage_counts' );
add_action( 'kkart_order_status_on-hold', 'kkart_update_coupon_usage_counts' );
add_action( 'kkart_order_status_cancelled', 'kkart_update_coupon_usage_counts' );

/**
 * Cancel all unpaid orders after held duration to prevent stock lock for those products.
 */
function kkart_cancel_unpaid_orders() {
	$held_duration = get_option( 'kkart_hold_stock_minutes' );

	if ( $held_duration < 1 || 'yes' !== get_option( 'kkart_manage_stock' ) ) {
		return;
	}

	$data_store    = KKART_Data_Store::load( 'order' );
	$unpaid_orders = $data_store->get_unpaid_orders( strtotime( '-' . absint( $held_duration ) . ' MINUTES', current_time( 'timestamp' ) ) );

	if ( $unpaid_orders ) {
		foreach ( $unpaid_orders as $unpaid_order ) {
			$order = kkart_get_order( $unpaid_order );

			if ( apply_filters( 'kkart_cancel_unpaid_order', 'checkout' === $order->get_created_via(), $order ) ) {
				$order->update_status( 'cancelled', __( 'Unpaid order cancelled - time limit reached.', 'kkart' ) );
			}
		}
	}
	wp_clear_scheduled_hook( 'kkart_cancel_unpaid_orders' );
	wp_schedule_single_event( time() + ( absint( $held_duration ) * 60 ), 'kkart_cancel_unpaid_orders' );
}
add_action( 'kkart_cancel_unpaid_orders', 'kkart_cancel_unpaid_orders' );

/**
 * Sanitize order id removing unwanted characters.
 *
 * E.g Users can sometimes try to track an order id using # with no success.
 * This function will fix this.
 *
 * @since 3.1.0
 * @param int $order_id Order ID.
 */
function kkart_sanitize_order_id( $order_id ) {
	return (int) filter_var( $order_id, FILTER_SANITIZE_NUMBER_INT );
}
add_filter( 'kkart_shortcode_order_tracking_order_id', 'kkart_sanitize_order_id' );

/**
 * Get an order note.
 *
 * @since  3.2.0
 * @param  int|WP_Comment $data Note ID (or WP_Comment instance for internal use only).
 * @return stdClass|null        Object with order note details or null when does not exists.
 */
function kkart_get_order_note( $data ) {
	if ( is_numeric( $data ) ) {
		$data = get_comment( $data );
	}

	if ( ! is_a( $data, 'WP_Comment' ) ) {
		return null;
	}

	return (object) apply_filters(
		'kkart_get_order_note',
		array(
			'id'            => (int) $data->comment_ID,
			'date_created'  => kkart_string_to_datetime( $data->comment_date ),
			'content'       => $data->comment_content,
			'customer_note' => (bool) get_comment_meta( $data->comment_ID, 'is_customer_note', true ),
			'added_by'      => __( 'Kkart', 'kkart' ) === $data->comment_author ? 'system' : $data->comment_author,
		),
		$data
	);
}

/**
 * Get order notes.
 *
 * @since  3.2.0
 * @param  array $args Query arguments {
 *     Array of query parameters.
 *
 *     @type string $limit         Maximum number of notes to retrieve.
 *                                 Default empty (no limit).
 *     @type int    $order_id      Limit results to those affiliated with a given order ID.
 *                                 Default 0.
 *     @type array  $order__in     Array of order IDs to include affiliated notes for.
 *                                 Default empty.
 *     @type array  $order__not_in Array of order IDs to exclude affiliated notes for.
 *                                 Default empty.
 *     @type string $orderby       Define how should sort notes.
 *                                 Accepts 'date_created', 'date_created_gmt' or 'id'.
 *                                 Default: 'id'.
 *     @type string $order         How to order retrieved notes.
 *                                 Accepts 'ASC' or 'DESC'.
 *                                 Default: 'DESC'.
 *     @type string $type          Define what type of note should retrieve.
 *                                 Accepts 'customer', 'internal' or empty for both.
 *                                 Default empty.
 * }
 * @return stdClass[]              Array of stdClass objects with order notes details.
 */
function kkart_get_order_notes( $args ) {
	$key_mapping = array(
		'limit'         => 'number',
		'order_id'      => 'post_id',
		'order__in'     => 'post__in',
		'order__not_in' => 'post__not_in',
	);

	foreach ( $key_mapping as $query_key => $db_key ) {
		if ( isset( $args[ $query_key ] ) ) {
			$args[ $db_key ] = $args[ $query_key ];
			unset( $args[ $query_key ] );
		}
	}

	// Define orderby.
	$orderby_mapping = array(
		'date_created'     => 'comment_date',
		'date_created_gmt' => 'comment_date_gmt',
		'id'               => 'comment_ID',
	);

	$args['orderby'] = ! empty( $args['orderby'] ) && in_array( $args['orderby'], array( 'date_created', 'date_created_gmt', 'id' ), true ) ? $orderby_mapping[ $args['orderby'] ] : 'comment_ID';

	// Set Kkart order type.
	if ( isset( $args['type'] ) && 'customer' === $args['type'] ) {
		$args['meta_query'] = array( // WPCS: slow query ok.
			array(
				'key'     => 'is_customer_note',
				'value'   => 1,
				'compare' => '=',
			),
		);
	} elseif ( isset( $args['type'] ) && 'internal' === $args['type'] ) {
		$args['meta_query'] = array( // WPCS: slow query ok.
			array(
				'key'     => 'is_customer_note',
				'compare' => 'NOT EXISTS',
			),
		);
	}

	// Set correct comment type.
	$args['type'] = 'order_note';

	// Always approved.
	$args['status'] = 'approve';

	// Does not support 'count' or 'fields'.
	unset( $args['count'], $args['fields'] );

	remove_filter( 'comments_clauses', array( 'KKART_Comments', 'exclude_order_comments' ), 10, 1 );

	$notes = get_comments( $args );

	add_filter( 'comments_clauses', array( 'KKART_Comments', 'exclude_order_comments' ), 10, 1 );

	return array_filter( array_map( 'kkart_get_order_note', $notes ) );
}

/**
 * Create an order note.
 *
 * @since  3.2.0
 * @param  int    $order_id         Order ID.
 * @param  string $note             Note to add.
 * @param  bool   $is_customer_note If is a costumer note.
 * @param  bool   $added_by_user    If note is create by an user.
 * @return int|WP_Error             Integer when created or WP_Error when found an error.
 */
function kkart_create_order_note( $order_id, $note, $is_customer_note = false, $added_by_user = false ) {
	$order = kkart_get_order( $order_id );

	if ( ! $order ) {
		return new WP_Error( 'invalid_order_id', __( 'Invalid order ID.', 'kkart' ), array( 'status' => 400 ) );
	}

	return $order->add_order_note( $note, (int) $is_customer_note, $added_by_user );
}

/**
 * Delete an order note.
 *
 * @since  3.2.0
 * @param  int $note_id Order note.
 * @return bool         True on success, false on failure.
 */
function kkart_delete_order_note( $note_id ) {
	return wp_delete_comment( $note_id, true );
}
