<?php
namespace Redaxscript;

use PDOException;
use function explode;
use function file_get_contents;
use function method_exists;
use function str_replace;
use function ucfirst;
use function version_compare;

/**
 * parent class to install the database
 *
 * @since 2.4.0
 *
 * @category Installer
 * @package Redaxscript
 * @author Henry Ruhs
 */

class Installer
{
	/**
	 * name of the directory
	 *
	 * @var string
	 */

	protected $_directory;

	/**
	 * placeholder for the prefix
	 *
	 * @var string
	 */

	protected $_prefixPlaceholder = '/* %PREFIX% */';

	/**
	 * constructor of the class
	 *
	 * @since 3.0.0
	 *
	 * @param Registry $_registry instance of the registry class
	 * @param Request $_request instance of the request class
	 * @param Language $_language instance of the language class
	 * @param Config $_config instance of the config class
	 */

	public function __construct(protected Registry $_registry, protected Request $_request, protected Language $_language, protected Config $_config)
	{
	}

	/**
	 * init the class
	 *
	 * @since 2.6.0
	 *
	 * @param string $directory name of the directory
	 */

	public function init(string $directory = 'database') : void
	{
		$this->_directory = $directory;
	}

	/**
	 * create from sql
	 *
	 * @since 2.4.0
	 *
	 * @return bool
	 */

	public function rawCreate() : bool
	{
		return $this->_rawExecute($this->_config->get('dbType'), 'create');
	}

	/**
	 * drop from sql
	 *
	 * @since 2.4.0
	 *
	 * @return bool
	 */

	public function rawDrop() : bool
	{
		return $this->_rawExecute($this->_config->get('dbType'), 'drop');
	}

	/**
	 * migrate from sql
	 *
	 * @since 4.4.0
	 *
	 * @param string $version version to migrate
	 *
	 * @return bool
	 */

	public function rawMigrate(string $version = null) : bool
	{
		return $this->_rawExecute($this->_config->get('dbType'), 'migrate', $version);
	}

	/**
	 * insert the data
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertData(array $optionArray = []) : void
	{
		$this->insertCategories($optionArray);
		$this->insertArticles($optionArray);
		$this->insertExtras($optionArray);
		$this->insertComments($optionArray);
		$this->insertGroups();
		$this->insertUsers($optionArray);
		$this->insertModules();
		$this->insertSettings($optionArray);
	}

	/**
	 * insert the categories
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertCategories(array $optionArray = []) : void
	{
		$now = $this->_registry->get('now');
		Db::forTablePrefix('categories')
			->create()
			->set(
			[
				'title' => 'Home',
				'alias' => 'home',
				'author' => $optionArray['adminName'],
				'rank' => 1,
				'date' => $now
			])
			->save();
	}

	/**
	 * insert the articles
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertArticles(array $optionArray = []) : void
	{
		$now = $this->_registry->get('now');
		Db::forTablePrefix('articles')
			->create()
			->set(
			[
				'title' => 'Welcome',
				'alias' => 'welcome',
				'author' => $optionArray['adminName'],
				'text' => file_get_contents('database' . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . 'articles' . DIRECTORY_SEPARATOR . 'welcome.phtml'),
				'category' => 1,
				'comments' => 1,
				'rank' => 1,
				'date' => $now
			])
			->save();
	}

	/**
	 * insert the extras
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertExtras(array $optionArray = []) : void
	{
		$now = $this->_registry->get('now');
		$extrasArray =
		[
			'categories' =>
			[
				'category' => null,
				'headline' => 1,
				'status' => 1
			],
			'articles' =>
			[
				'category' => null,
				'headline' => 1,
				'status' => 1
			],
			'comments' =>
			[
				'category' => null,
				'headline' => 1,
				'status' => 1
			],
			'languages' =>
			[
				'category' => null,
				'headline' => 1,
				'status' => 0
			],
			'templates' =>
			[
				'category' => null,
				'headline' => 1,
				'status' => 0
			],
			'teaser' =>
			[
				'category' => 1,
				'headline' => 0,
				'status' => 0
			]
		];
		$extrasRank = 0;

		/* process extras */

		foreach ($extrasArray as $key => $value)
		{
			Db::forTablePrefix('extras')
				->create()
				->set(
				[
					'title' => ucfirst($key),
					'alias' => $key,
					'author' => $optionArray['adminName'],
					'text' => file_get_contents('database' . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . 'extras' . DIRECTORY_SEPARATOR . $key . '.phtml'),
					'category' => $value['category'],
					'headline' => $value['headline'],
					'status' => $value['status'],
					'rank' => ++$extrasRank,
					'date' => $now
				])
				->save();
		}
	}

	/**
	 * insert the comments
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertComments(array $optionArray = []) : void
	{
		$now = $this->_registry->get('now');
		Db::forTablePrefix('comments')
			->create()
			->set(
			[
				'author' => $optionArray['adminName'],
				'email' => $optionArray['adminEmail'],
				'text' => file_get_contents('database' . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . 'comments' . DIRECTORY_SEPARATOR . 'hello.phtml'),
				'article' => 1,
				'rank' => 1,
				'date' => $now
			])
			->save();
	}

	/**
	 * insert the groups
	 *
	 * @since 3.1.0
	 */

	public function insertGroups() : void
	{
		Db::forTablePrefix('groups')
			->create()
			->set(
			[
				'name' => 'Administrators',
				'alias' => 'administrators',
				'description' => 'Unlimited access',
				'categories' => '[1, 2, 3]',
				'articles' => '[1, 2, 3]',
				'extras' => '[1, 2, 3]',
				'comments' => '[1, 2, 3]',
				'groups' => '[1, 2, 3]',
				'users' => '[1, 2, 3]',
				'modules' => '[1, 2, 3]',
				'settings' => 1,
				'filter' => 0
			])
			->save();
		Db::forTablePrefix('groups')
			->create()
			->set(
			[
				'name' => 'Members',
				'alias' => 'members',
				'description' => 'Default members group'
			])
			->save();
	}

	/**
	 * insert the users
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertUsers(array $optionArray = []) : void
	{
		$passwordHash = new Hash();
		$passwordHash->init($optionArray['adminPassword']);
		Db::forTablePrefix('users')
			->create()
			->set(
			[
				'name' => $optionArray['adminName'],
				'user' => $optionArray['adminUser'],
				'password' => $passwordHash->getHash(),
				'email' => $optionArray['adminEmail'],
				'description' => 'God admin',
				'groups' => '[1]'
			])
			->save();
	}

	/**
	 * insert the modules
	 *
	 * @since 3.1.0
	 */

	public function insertModules() : void
	{
		$moduleArray =
		[
			'AliasGenerator',
			'CallHome',
			'ConfirmAction',
			'FormValidator',
			'RankSorter',
			'TextareaResizer',
			'UnmaskPassword',
			'VisualEditor'
		];

		/* process modules */

		foreach ($moduleArray as $alias)
		{
			$moduleClass = 'Redaxscript\Modules\\' . $alias . '\\' . $alias;

			/* install */

			if (method_exists($moduleClass, 'install'))
			{
				$module = new $moduleClass($this->_registry, $this->_request, $this->_language, $this->_config);
				$module->install();
			}
		}
	}

	/**
	 * insert the settings
	 *
	 * @since 3.1.0
	 *
	 * @param array $optionArray options of the installation
	 */

	public function insertSettings(array $optionArray = []) : void
	{
		$settingArray =
		[
			'language' => null,
			'template' => null,
			'title' => $this->_language->get('_package')['name'],
			'author' => $optionArray['adminName'],
			'copyright' => null,
			'description' => $this->_language->get('_package')['description'],
			'keywords' => null,
			'robots' => 1,
			'email' => $optionArray['adminEmail'],
			'smtp' => null,
			'subject' => $this->_language->get('_package')['name'],
			'notification' => 0,
			'charset' => 'UTF-8',
			'locale' => 'en_US',
			'divider' => ' - ',
			'zone' => 'Europe/Berlin',
			'time' => 'H:i',
			'date' => 'd.m.Y',
			'homepage' => 0,
			'limit' => 10,
			'order' => 'asc',
			'pagination' => 1,
			'registration' => 0,
			'verification' => 0,
			'recovery' => 1,
			'moderation' => 0,
			'captcha' => 0,
			'version' => $this->_language->get('_package')['version']
		];

		/* process settings */

		foreach ($settingArray as $name => $value)
		{
			Db::forTablePrefix('settings')
				->create()
				->set(
				[
					'name' => $name,
					'value' => $value
				])
				->save();
		}
	}

	/**
	 * execute from sql
	 *
	 * @since 4.5.0
	 *
	 * @param string $type type of the database
	 * @param string $action action to process
	 * @param string $version version to process
	 *
	 * @return bool
	 */

	protected function _rawExecute(string $type = null, string $action = null, string $version = null) : bool
	{
		$actionFilesystem = new Filesystem\File();
		$actionFilesystem->init($this->_directory . DIRECTORY_SEPARATOR . $type . DIRECTORY_SEPARATOR . $action);
		$actionFilesystemArray = $actionFilesystem->getSortArray();
		$dbPrefix = $this->_config->get('dbPrefix');
		$status = true;

		/* process filesystem */

		foreach ($actionFilesystemArray as $file)
		{
			$query = $this->_validateFileByVersion($file, $version) ? $actionFilesystem->readFile($file) : null;
			if ($query)
			{
				if ($dbPrefix)
				{
					$query = str_replace($this->_prefixPlaceholder, $dbPrefix, $query);
				}
				try
				{
					Db::rawExecute($query);
				}
				catch (PDOException $exception)
				{
					$status = false;
				}
			}
		}
		return $status;
	}

	/**
	 * validate file by version
	 *
	 * @since 4.5.0
	 *
	 * @param string $file name of the file
	 * @param string $version version to process
	 *
	 * @return bool
	 */

	protected function _validateFileByVersion(string $file = null, string $version = null) : bool
	{
		$fileArray = explode('_', $file);
		return version_compare($version, $fileArray[0], '<');
	}
}
