<?php
 /**
 * Jamroom System Tools module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2017 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrSystemTools_meta()
{
    return array(
        'name'        => 'System Tools',
        'url'         => 'systemtools',
        'version'     => '2.0.1',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Provides low level command line tools used by modules',
        'category'    => 'core',
        'requires'    => 'jrCore:6.5.12',
        'license'     => 'mpl',
        'locked'      => true,
        'activate'    => true
    );
}

/**
 * init
 */
function jrSystemTools_init()
{
    // Make sure tools actually work
    jrCore_register_event_listener('jrCore', 'system_check', 'jrSystemTools_system_check_listener');

    // Our default view for admins
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrSystemTools', 'admin/info');

    return true;
}

//--------------------------
// EVENT LISTENERS
//--------------------------

/**
 * Check that bundled binaries are executable
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrSystemTools_system_check_listener($_data, $_user, $_conf, $_args, $event)
{
    foreach (jrSystemTools_get_tools() as $tool => $_inf) {
        $dat             = array();
        $dat[1]['title'] = $tool;
        $dat[1]['class'] = 'center';
        $dat[2]['title'] = 'executable';
        $dat[2]['class'] = 'center';
        // Make sure we are installed
        $mod = null;
        if (isset($_inf['alternate'])) {
            $mod = $_inf['alternate'];
        }
        if (!$cmd = jrCore_get_tool_path($tool, $mod)) {
            $dat[3]['title'] = $_args['fail'];
            $dat[4]['title'] = "{$tool} binary is not executable";
        }
        else {
            // Make sure it actually works
            if (jrSystemTools_test_tool($tool)) {
                // We're good
                $dat[3]['title'] = $_args['pass'];
                $dat[4]['title'] = "{$tool} binary is working correctly";
            }
            else {
                $dat[3]['title'] = $_args['fail'];
                $dat[4]['title'] = str_replace(APP_DIR . '/', '', $cmd) . ' is not working correctly';
            }
        }
        $dat[3]['class'] = 'center';
        jrCore_page_table_row($dat);
    }
    return $_data;
}

//--------------------------
// FUNCTIONS
//--------------------------

/**
 * Test a system tool to see if it is working
 * @param string $tool
 * @param string $full_path set to full path of tool
 * @return bool
 */
function jrSystemTools_test_tool($tool, $full_path = null)
{
    if ($_tools = jrSystemTools_get_tools()) {
        if (isset($_tools[$tool])) {
            $mod = null;
            if (isset($_tools[$tool]['alternate'])) {
                $mod = $_tools[$tool]['alternate'];
            }
            if (is_null($full_path)) {
                $cmd = jrCore_get_tool_path($tool, $mod);
            }
            else {
                $cmd = $full_path;
            }
            if (!is_executable($cmd)) {
                return false;
            }
            else {
                ob_start();
                jrCore_run_timed_command("{$cmd} {$_tools[$tool]['command']}", 3, $stdout, $stderr);
                ob_end_clean();
                if (!empty($stdout) && strpos(' ' . $stdout, $_tools[$tool]['result']) && !stripos($stdout, 'No such file')) {
                    return true;
                }
                if (!empty($stderr) && strpos(' ' . $stderr, $_tools[$tool]['result']) && !stripos($stderr, 'No such file')) {
                    return true;
                }
            }
        }
    }
    return false;
}

/**
 * Get all the paths on a server a tool can be found at
 * @param string $tool Tool (binary name)
 * @return array
 */
function jrSystemTools_get_all_paths_for_tool($tool)
{
    $_paths = array(
        '/usr/bin'                               => '/usr/bin',
        '/usr/sbin'                              => '/usr/sbin',
        '/usr/local/bin'                         => '/usr/local/bin',
        '/usr/local/sbin'                        => '/usr/local/sbin',
        '/opt/homebrew/bin'                      => '/opt/homebrew/bin',
        APP_DIR . '/modules/jrSystemTools/tools' => 'jrSystemTools/tools'
    );
    $_tools = array('auto' => 'Auto Detect');
    foreach ($_paths as $path => $disp) {
        if (is_file("{$path}/{$tool}") && is_executable("{$path}/{$tool}")) {
            $_tools["{$path}/{$tool}"] = "{$disp}/{$tool}";
        }
    }
    return $_tools;
}

/**
 * Get provided tools
 * @return array
 */
function jrSystemTools_get_tools()
{
    return array(
        'convert'   => array(
            'command'   => '-version',
            'result'    => 'ImageMagick',
            'alternate' => 'jrImage',
            'sublabel'  => 'resizes animated GIFs',
            'help'      => 'The <b>convert</b> tool is used to resize animated GIF images'
        ),
        'ffmpeg'    => array(
            'command'   => '-version',
            'result'    => 'libavutil',
            'alternate' => 'jrCore',
            'sublabel'  => 'converts audio and video',
            'help'      => 'The <b>ffmpeg</b> tool is used for converting Audio and Video files in to a format that the system can use'
        ),
        'id3v2'     => array(
            'command'   => '-h',
            'result'    => '[OPTION]',
            'alternate' => 'jrAudio',
            'sublabel'  => 'read and write audio ID3 tags',
            'help'      => 'The <b>id3v2</b> tool is used to read ID3 tags that are found in some audio files'
        ),
        'jpegoptim' => array(
            'command'  => '--version',
            'result'   => 'Copyright',
            'sublabel' => 'optimizes JPEG images',
            'help'     => 'The <b>jpegoptim</b> tool is used to optimize JPG images'
        ),
        'sox'       => array(
            'command'   => '--version',
            'result'    => 'SoX',
            'alternate' => 'jrAudio',
            'sublabel'  => 'audio file editing tool',
            'help'      => 'The <b>sox</b> tool is used to manipulate audio files and provide fade-in and fade-out functionality when creating audio snippets',
        ),
        'tifig'     => array(
            'command'  => '-i',
            'result'   => 'missing an argument',
            'sublabel' => 'convert HEIC images to JPG',
            'help'     => 'The <b>tifig</b> tool is used to convert HEIC and HEIF image formats to JPG',
        ),
        'wget'      => array(
            'command'   => '--version',
            'result'    => 'Wget',
            'alternate' => 'jrUrlScan',
            'sublabel'  => 'command line HTML browser',
            'help'      => 'The <b>wget</b> tool is used to download web pages and parse HTML for Open Graph tags',
        )
    );
}

