<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrProfile_meta()
{
    return array(
        'name'        => 'Profiles',
        'url'         => 'profile',
        'version'     => '2.0.1',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Core support for Profiles and Profile Quotas',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/955/profiles',
        'category'    => 'profiles',
        'license'     => 'mpl',
        'priority'    => 200, // lower priority so we can pick up other modules additions
        'requires'    => 'jrCore:6.5.12,jrUser:2.6.0',
        'locked'      => true,
        'activate'    => true
    );
}

/**
 * init
 */
function jrProfile_init()
{
    // We have some JS and CSS
    jrCore_register_module_feature('jrCore', 'css', 'jrProfile', 'jrProfile.css');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrProfile', 'jrProfile.js');

    // Let modules get into the profile data
    jrCore_register_event_trigger('jrProfile', 'get_profile_info', 'Fired when information about a specific profile is requested');
    jrCore_register_event_trigger('jrProfile', 'get_pulse_keys', 'Fired when pulse key counts are retrieved');
    jrCore_register_event_trigger('jrProfile', 'reset_pulse_key', 'Fired when a pulse key is reset');
    jrCore_register_event_trigger('jrProfile', 'profile_data', 'Fired with profile data when a profile page is viewed');
    jrCore_register_event_trigger('jrProfile', 'profile_view', 'Fired when a profile page is viewed');
    jrCore_register_event_trigger('jrProfile', 'profile_index', 'Fired when the profile index is viewed');
    jrCore_register_event_trigger('jrProfile', 'item_index_view', 'Fired when a profile index page for a module is viewed');
    jrCore_register_event_trigger('jrProfile', 'item_list_view', 'Fired when a profile list page for a module is viewed');
    jrCore_register_event_trigger('jrProfile', 'item_detail_view', 'Fired when a specific item detail page is viewed');
    jrCore_register_event_trigger('jrProfile', 'item_module_tabs', 'Fired when a tab bar is created for a module profile view');
    jrCore_register_event_trigger('jrProfile', 'delete_profile_init', 'Fired before a profile and its data are deleted');
    jrCore_register_event_trigger('jrProfile', 'delete_profile', 'Fired after a profile and its data are deleted');
    jrCore_register_event_trigger('jrProfile', 'profile_menu_params', 'Fired when a profile menu is viewed');
    jrCore_register_event_trigger('jrProfile', 'change_active_profile', 'Fired when a users active profile is changed');
    jrCore_register_event_trigger('jrProfile', 'user_linked_profile_ids', 'Fired when gathering linked profile_ids for a user_id');
    jrCore_register_event_trigger('jrProfile', 'profile_browser_row', 'Fired with profile info for each row in the Profile Browser');

    // Listen for user sign ups so we can create associated profile
    jrCore_register_event_listener('jrUser', 'signup_created', 'jrProfile_signup_created_listener', true);
    jrCore_register_event_listener('jrUser', 'signup_activated', 'jrProfile_signup_activated_listener', true);

    // Core Event Listeners
    jrCore_register_event_listener('jrCore', 'db_increment_key', 'jrProfile_db_increment_key_listener');
    jrCore_register_event_listener('jrCore', 'db_search_params', 'jrProfile_db_search_params_listener');
    jrCore_register_event_listener('jrCore', 'db_search_items', 'jrProfile_db_search_items_listener');
    jrCore_register_event_listener('jrCore', 'db_search_count_query', 'jrProfile_db_search_count_query_listener');
    jrCore_register_event_listener('jrCore', 'db_delete_item', 'jrProfile_db_delete_item_listener');
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrProfile_verify_module_listener');
    jrCore_register_event_listener('jrCore', 'repair_module', 'jrProfile_repair_module_listener');
    jrCore_register_event_listener('jrCore', 'process_exit', 'jrProfile_process_exit_listener');
    jrCore_register_event_listener('jrCore', 'process_init', 'jrProfile_process_init_listener');
    jrCore_register_event_listener('jrCore', 'expire_recycle_bin', 'jrProfile_expire_recycle_bin_listener');
    jrCore_register_event_listener('jrCore', 'save_media_file', 'jrProfile_save_media_file_listener');
    jrCore_register_event_listener('jrCore', 'hourly_maintenance', 'jrProfile_hourly_maintenance_listener');
    jrCore_register_event_listener('jrProfile', 'item_index_view', 'jrProfile_item_index_view_listener');

    // Tell the comment module our private profiles
    jrCore_register_event_listener('jrComment', 'private_item_ids', 'jrProfile_private_item_ids_listener');

    // Register our tools
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrProfile', 'quota_browser', array('Quota Browser', 'Browse the existing Profile Quotas in your system'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrProfile', 'quota_compare', array('Quota Compare', 'Compare Modules allowed in each profile quota'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrProfile', 'create', array('Create New Profile', 'Create a new Profile'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrProfile', 'user_link', array('Link User Accounts', 'Link User Accounts with existing Profiles'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrProfile', 'rebuild_quota_counts', array('Rebuild Profile Counts', 'Rebuild the counts for Profiles in each Quota'));
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrProfile', 'quota_browser', 'Quota Browser');

    // Register our account tab..
    jrCore_register_module_feature('jrUser', 'account_tab', 'jrProfile', 'settings', 2);

    // Allow admin to customize our forms
    jrCore_register_module_feature('jrCore', 'designer_form', 'jrProfile', 'create');
    jrCore_register_module_feature('jrCore', 'designer_form', 'jrProfile', 'settings');

    // We provide our own data browser
    jrCore_register_module_feature('jrCore', 'data_browser', 'jrProfile', 'jrProfile_data_browser');

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrProfile', 'profile_link', 'user_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_profile_id_table', 'jrProfile', 'profile_link', 'profile_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_profile_id_table', 'jrProfile', 'pulse', 'pulse_profile_id');

    // We have fields that can be search
    jrCore_register_module_feature('jrSearch', 'search_fields', 'jrProfile', 'profile_name', 26);

    // Link to manage profiles (Power Users and Multi Profile users)
    $_tmp = array(
        'group'    => 'power,multi',
        'label'    => 25,
        'url'      => 'list_profiles',
        'function' => 'jrProfile_more_than_one_linked_profile'
    );
    jrCore_register_module_feature('jrCore', 'skin_menu_item', 'jrProfile', 'profile_manage', $_tmp);

    // We provide some dashboard panels
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrProfile', 'total profiles', 'jrProfile_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrProfile', 'signups today', 'jrProfile_dashboard_panels');

    // Graph Support
    $_tmp = array(
        'title'    => 'Profiles Created by Day',
        'function' => 'jrProfile_graph_profiles_by_day',
        'group'    => 'admin'
    );
    jrCore_register_module_feature('jrGraph', 'graph_config', 'jrProfile', 'profiles_by_day', $_tmp);

    $_tmp = array(
        'title'    => 'Signups by Day',
        'function' => 'jrProfile_graph_signups_by_day',
        'group'    => 'admin'
    );
    jrCore_register_module_feature('jrGraph', 'graph_config', 'jrProfile', 'signups_by_day', $_tmp);

    // Akismet Anti Spam support
    jrCore_register_module_feature('jrAkismet', 'spam_check', 'jrProfile', 'comment', 'reject');

    // Keep track of disk usage
    jrCore_register_queue_worker('jrProfile', 'update_profile_disk_usage', 'jrProfile_update_profile_disk_usage_worker', 0, 2, 300, LOW_PRIORITY_QUEUE);

    // Core magic views
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrProfile', 'rebuild_count', 'view_jrProfile_rebuild_count');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrProfile', 'rebuild_count_save', 'view_jrProfile_rebuild_count_save');

    // Register this tool for every module that has a datastore (not profile, user, forum or group page)
    if ($_tmp = jrProfile_get_valid_modules()) {
        foreach ($_tmp as $mod => $name) {
            jrCore_register_module_feature('jrCore', 'tool_view', $mod, 'rebuild_count', array('Rebuild Profile Counts', "Rebuild the profile item counts for {$name} items"));
        }
    }

    return true;
}

// Bring in Library
require_once APP_DIR . '/modules/jrProfile/lib/events.php';
require_once APP_DIR . '/modules/jrProfile/lib/functions.php';
require_once APP_DIR . '/modules/jrProfile/lib/quota.php';
require_once APP_DIR . '/modules/jrProfile/lib/smarty.php';

//-------------------------
// QUEUE WORKER
//-------------------------

/**
 * Update the disk usage for an array of profile_ids
 * @param array $_queue
 * @return bool
 */
function jrProfile_update_profile_disk_usage_worker($_queue)
{
    if (jrCore_get_config_value('jrProfile', 'disable_profile_disk_usage', 'off') == 'off') {
        if (isset($_queue['profile_ids']) && is_array($_queue['profile_ids'])) {
            $_up = array();
            foreach ($_queue['profile_ids'] as $pid) {
                if ($pid > 0) {
                    $_up[$pid] = array(
                        'profile_disk_usage' => jrProfile_get_disk_usage($pid)
                    );
                }
            }
            if (count($_up) > 0) {
                jrCore_db_update_multiple_items('jrProfile', $_up);
            }
        }
    }
    return true;
}

//-------------------------
// DASHBOARD
//-------------------------

/**
 * User Profiles Dashboard Panels
 * @param $panel
 * @return array|false
 */
function jrProfile_dashboard_panels($panel)
{
    // The panel being asked for will come in as $panel
    $out = false;
    switch ($panel) {

        case 'total profiles':
            $out = array(
                'title' => jrCore_number_format(jrCore_db_get_ds_row_count('jrProfile')),
                'graph' => 'profiles_by_day'
            );
            break;

        case 'signups today':
            $key = 'dashboard_panel_signups_today';
            $num = jrCore_is_cached('jrProfile', $key, false, false, true, false);
            if (!is_numeric($num)) {
                $beg = strtotime('midnight', time());
                $num = jrCore_db_run_key_function('jrProfile', '_created', "> {$beg}", 'COUNT');
                jrCore_add_to_cache('jrProfile', $key, $num, 600, 0, false, false, false);
            }
            $out = array(
                'title' => jrCore_number_format($num),
                'graph' => 'signups_by_day'
            );
            break;

    }
    return ($out) ? $out : false;
}

//-------------------------
// GRAPHS
//-------------------------

/**
 * Profiles created by day
 * @param $module string Module
 * @param $name string Name of Graph to create
 * @param $_args array Passed in Parameters
 * @return array
 */
function jrProfile_graph_profiles_by_day($module, $name, $_args)
{
    $_rs = array(
        '_sets' => array(
            0 => array(
                'label'       => "Profiles Created",
                'date_format' => '%m/%d/%Y',
                'minTickSize' => "[1, 'day']",
                'type'        => 'line',
                'pointRadius' => 3,
                '_data'       => array(),
            )
        )
    );

    // Get our data
    $old = (time() - (60 * 86400));
    $off = date_offset_get(new DateTime);
    if (jrCore_db_key_has_index_table('jrProfile', '_created')) {
        $tbl = jrCore_db_get_index_table_name('jrProfile', '_created');
        $req = "SELECT FROM_UNIXTIME((`value` + {$off}), '%Y%m%d') AS c, COUNT(`_item_id`) AS n FROM {$tbl} WHERE `value` > {$old} GROUP BY c ORDER BY c ASC LIMIT 60";
    }
    else {
        $tbl = jrCore_db_table_name('jrProfile', 'item_key');
        $req = "SELECT FROM_UNIXTIME((`value` + {$off}), '%Y%m%d') AS c, COUNT(`_item_id`) AS n FROM {$tbl} WHERE (`key` = '_created' AND `value` > {$old}) GROUP BY c ORDER BY c ASC LIMIT 60";
    }
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $v) {
            $yr = substr($v['c'], 0, 4);
            $mn = substr($v['c'], 4, 2);
            $dy = substr($v['c'], 6, 2);
            $tm = (string) mktime(0, 0, 0, $mn, $dy, $yr);
            if (!isset($_rs['_sets'][0]['_data']["{$tm}"])) {
                $_rs['_sets'][0]['_data']["{$tm}"] = 0;
            }
            $_rs['_sets'][0]['_data']["{$tm}"] += $v['n'];
        }
    }
    return $_rs;
}

/**
 * Signups (user accounts) created by day
 * @param $module string Module
 * @param $name string Name of Graph to create
 * @param $_args array Passed in Parameters
 * @return array
 */
function jrProfile_graph_signups_by_day($module, $name, $_args)
{
    $_rs = array(
        '_sets' => array(
            0 => array(
                'label'       => "Accounts Created",
                'date_format' => '%m/%d/%Y',
                'minTickSize' => "[1, 'day']",
                'type'        => 'line',
                'pointRadius' => 3,
                '_data'       => array(),
            )
        )
    );

    // Get our data
    $old = (time() - (60 * 86400));
    $off = date_offset_get(new DateTime);
    $tbl = jrCore_db_table_name('jrProfile', 'item_key');
    $req = "SELECT FROM_UNIXTIME((`value` + {$off}), '%Y%m%d') AS c, COUNT(`_item_id`) AS n FROM {$tbl} WHERE (`key` = '_created' AND `value` > {$old}) GROUP BY c ORDER BY c ASC LIMIT 60";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $v) {
            $yr = substr($v['c'], 0, 4);
            $mn = substr($v['c'], 4, 2);
            $dy = substr($v['c'], 6, 2);
            $tm = (string) mktime(0, 0, 0, $mn, $dy, $yr);
            if (!isset($_rs['_sets'][0]['_data']["{$tm}"])) {
                $_rs['_sets'][0]['_data']["{$tm}"] = 0;
            }
            $_rs['_sets'][0]['_data']["{$tm}"] += $v['n'];
        }
    }
    return $_rs;
}
