<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Page Elements
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Shows a notice message and is used when we are NOT sure if the
 * UI is even working.  Will show a plain text error message
 * @param string $type Error level (CRI, MAJ, etc.)
 * @param string $message Error Message
 * @param bool $template set to FALSE for no template
 * @return null
 */
function jrCore_notice($type, $message, $template = true)
{
    jrCore_db_close();
    // @note: We don't run process_exit here since we could be in a situation
    // where we are overloaded, down, etc. and we dont want to spin up workers
    if (jrCore_is_ajax_request()) {
        $_out              = array('notices' => array());
        $_out['notices'][] = array('type' => strtolower($type), 'text' => $message);
        return jrCore_json_response($_out, true, true, false);
    }
    if ($template) {
        $_rp = array(
            'type'    => $type,
            'message' => $message
        );
        $out = jrCore_parse_template('error_notice.tpl', $_rp, 'jrCore');
    }
    else {
        $out = $message;
    }
    jrCore_send_response_and_detach($out, false);
    exit;  // OK - not called but here to see that jrCore_notice exits
}

/**
 * Show a success/warning/error notice page and Exit
 * @note function exits
 * @param string $notice_type Notice Type (success/warning/notice/error)
 * @param string $notice_text Notice Text
 * @param string $cancel_url URL to link to Cancel Button
 * @param string $cancel_text Text for Cancel Button
 * @param bool $clean_output If true, Notice Text is run through htmlspecialchars()
 * @param bool $include_header If true, header/footer is included in output
 * @return null
 */
function jrCore_notice_page($notice_type, $notice_text, $cancel_url = null, $cancel_text = null, $clean_output = true, $include_header = true)
{
    global $_post;
    if (jrCore_checktype($notice_text, 'number_nz')) {
        $_lang = jrUser_load_lang_strings();
        if (isset($_lang["{$_post['module']}"][$notice_text])) {
            $notice_text = $_lang["{$_post['module']}"][$notice_text];
        }
    }

    if (php_sapi_name() === 'cli') {
        echo "{$notice_type} Error: {$notice_text}\n";
        exit(1); // OK
    }

    if (jrCore_is_ajax_request()) {
        $_er = array($notice_type => $notice_text);
        return jrCore_json_response($_er);
    }

    jrCore_page_title($notice_type);
    jrCore_page_notice($notice_type, $notice_text, $clean_output);

    if (!$include_header) {
        jrCore_page_set_no_header_or_footer();
    }
    else {
        jrCore_delete_flag('jrcore_page_no_header_or_footer');
    }
    if (!empty($cancel_url)) {
        jrCore_page_cancel_button($cancel_url, $cancel_text);
    }

    $out = jrCore_page_display(true);
    $out = jrCore_trigger_event('jrCore', 'view_results', $out);

    jrCore_send_response_and_detach($out);
    exit;
}

/**
 * Add a page element to the jrcore_page_elements flag for processing at view time.
 * @param string $section section string section to add to: "meta", "javascript", "css", "page", "footer"
 * @param array $_params Element information as an array to pass to the element's template
 * @return bool
 */
function jrCore_create_page_element($section, $_params)
{
    global $_post, $_mods;
    $_tmp = jrCore_get_flag('jrcore_page_elements');
    if (!$_tmp || !is_array($_tmp)) {
        $_tmp = array();
    }
    switch ($section) {
        case 'meta':
            // Meta data is simple key value pairs.
            if (!isset($_tmp[$section])) {
                $_tmp[$section] = array();
            }
            foreach ($_params as $key => $value) {
                $_tmp[$section][$key] = $value;
            }
            break;
        case 'css_embed':
        case 'javascript_embed':
        case 'javascript_ready_function':
        case 'javascript_footer_function':
            if (!isset($_tmp[$section])) {
                $_tmp[$section] = '';
            }
            foreach ($_params as $entry) {
                // See if we are including lightbox
                if (strpos($entry, 'lightbox')) {
                    jrCore_set_flag('jrcore_lightbox_included', 1);
                }
                $_tmp[$section] .= "{$entry}\n";
            }
            break;
        case 'javascript_header_href':
        case 'javascript_footer_href':
        case 'javascript_href':
        case 'css_footer_href':
        case 'css_href':
            if (!isset($_tmp[$section])) {
                $_tmp[$section] = array();
            }
            foreach ($_params as $k => $prm) {
                if ($k === 'source' && !strpos($prm, '?')) {
                    if (isset($_post['module'])) {
                        if (isset($_mods["{$_post['module']}"]['module_version']) && (!isset($_params['include_version']) || jrCore_checktype($_params['include_version'], 'is_true'))) {
                            $_params[$k] = "{$prm}?v={$_mods["{$_post['module']}"]['module_version']}";
                        }
                        else {
                            $_params[$k] = $prm;
                        }
                    }
                }
            }
            $_tmp[$section][] = $_params;
            break;

        // BELOW USED INTERNALLY - do not use in view controllers.
        case 'form_begin':
            // Only one form per page
            $_tmp['form_begin'] = $_params['form_html'];
            // We also add in to regular page elements so the item templates
            // will be rendered in the correct place
            $_tmp['page'][] = $_params;
            break;
        case 'form_end':
            $_tmp[$section] = $_params['form_html'];
            break;
        case 'form_hidden':
            if (!isset($_tmp[$section])) {
                $_tmp[$section] = array();
            }
            $_tmp[$section][] = $_params['form_html'];
            break;
        case 'form_modal':
            $_tmp['form_modal'] = $_params;
            break;

        // "page" is default
        default:
            if (empty($_params['type'])) {
                jrCore_logger('CRI', "core: required element type not received - verify usage");
                return false;
            }
            if (!isset($_tmp[$section])) {
                $_tmp[$section] = array();
            }
            $_tmp[$section][] = $_params;
            break;
    }
    return jrCore_set_flag('jrcore_page_elements', $_tmp);
}

/**
 * Convert applicable strings to HTML entities
 * @param $string string String to convert
 * @return string
 */
function jrCore_entity_string($string)
{
    return (is_string($string)) ? htmlentities($string, ENT_QUOTES, 'UTF-8', false) : $string;
}

/**
 * Escape a string so it can be used in JS
 * @param string $string
 * @return string
 */
function jrCore_escape_js_string($string)
{
    return str_replace(array("\n\r", "\n", "\r"), '\\n', addslashes($string));
}

/**
 * Highlight a string by adding the page_search_highlight class to a surrounding span
 * @param string $string String to be hilighted
 * @param string $search Sub-String within String to be hilighted
 * @return string
 */
function jrCore_hilight_string($string, $search)
{
    // Does our string have HTML in it?
    if (strpos($string, '>')) {
        // We have HTML in our string - we only highlight sections of text that are NOT inside HTML tag quotes
        // <b>Highlight in this string is OK</b><a title="But not in this one">And here is good too</a>
        preg_match_all('#<[^>]+>(.+?)#ims', $string, $_matches);
        if ($_matches && is_array($_matches[0])) {
            $temp = jrCore_entity_string(jrCore_strip_html($search));
            $_tmp = array_flip($_matches[0]);
            foreach ($_tmp as $html => $key) {
                $string = str_replace($html, "%%{$key}%%", $string);
            }
            $string = str_ireplace($search, '<span class="page_search_highlight">' . $temp . '</span>', $string);
            foreach ($_tmp as $html => $key) {
                $string = str_replace("%%{$key}%%", $html, $string);
            }
            return $string;
        }
    }
    $search = preg_quote($search, '/');
    return preg_replace("/({$search})/i", '<span class="page_search_highlight">$1</span>', $string);
}

/**
 * set HTML page title
 * @param string $title Title of page
 * @param bool $overwrite set to false to prevent later calls overwriting earlier
 * @return bool
 */
function jrCore_page_title($title, $overwrite = true)
{
    $title = html_entity_decode(jrCore_strip_html($title), ENT_QUOTES);
    if ($overwrite) {
        jrCore_set_flag('jrcore_html_page_title', $title);
    }
    else {
        if (!jrCore_get_flag('jrcore_html_page_title')) {
            jrCore_set_flag('jrcore_html_page_title', $title);
        }
    }
    return true;
}

/**
 * Create a "page jumper" select element
 * @param string $dom_id DOM ID of select element
 * @param array $_options Select options
 * @param string $selected Pre-selected Value
 * @param string $onchange Onchange JS
 * @return string
 */
function jrCore_page_jumper($dom_id, $_options, $selected, $onchange, $class = '')
{
    $_rep = array(
        'dom_id'   => $dom_id,
        '_options' => $_options,
        'selected' => $selected,
        'onchange' => $onchange,
        'class'    => $class
    );
    return trim(jrCore_parse_template('page_jumper.tpl', $_rep, 'jrCore'));
}

/**
 * a page jumper select used in a form page banner
 * @param string $module Module Name
 * @param string $field Field to link to ID
 * @param array $search Search parameters for jrCore_db_search_items
 * @param string $create Create View for module
 * @param string $update Update View for module
 * @return string
 */
function jrCore_page_banner_item_jumper($module, $field, $search, $create, $update)
{
    global $_post;
    if (empty($search) || !is_array($search)) {
        return false;
    }
    $_rt = false;
    if (count($search) === 1) {
        // See if this is a _profile_id = search
        $temp = reset($search);
        if (is_string($temp) && strpos(trim($temp), '_profile_id =') === 0) {
            list(, $pid) = explode('=', $temp);
            $pid = intval($pid);
            if (!empty($pid)) {
                if ($_rt = jrCore_db_get_all_values_for_key_by_profile_id($pid, $module, $field)) {
                    natcasesort($_rt);
                }
                else {
                    $_rt = array();
                }
            }
        }
    }
    if ($_rt === false) {
        $_sc = array(
            'search'         => $search,
            'order_by'       => array($field => 'asc'),
            'return_keys'    => array('_item_id', $field),
            'skip_triggers'  => true,
            'privacy_check'  => false,
            'index_key'      => '_item_id',
            'ignore_pending' => true,
            'limit'          => 500
        );
        if ($_sc = jrCore_db_search_items($module, $_sc)) {
            $_rt = $_sc['_items'];
        }
        unset($_sc);
    }

    $_options = array('' => '-');
    if (strlen($create) > 0) {
        $_ln                = jrUser_load_lang_strings();
        $_options['create'] = $_ln['jrCore'][50];
    }
    if (!empty($_rt)) {
        foreach ($_rt as $k => $v) {
            $_options[$k] = $v;
        }
    }

    $c_url   = jrCore_get_base_url() . "/{$_post['module_url']}/{$create}";
    $u_url   = jrCore_get_base_url() . "/{$_post['module_url']}/{$update}/id=";
    $onclick = "var i=$(this).val(); if (i == 'create') { jrCore_window_location('{$c_url}') } else if (Number(i) > 0) { jrCore_window_location('{$u_url}' + i) }";
    return jrCore_page_jumper('jumper_item_id', $_options, $_post['id'], $onclick);
}

/**
 * Show a "banner" at the top of a form page
 * @param string $title Title of section
 * @param string $subtitle Subtitle text for section
 * @param string $icon Icon image
 * @return bool
 */
function jrCore_page_banner($title, $subtitle = null, $icon = null)
{
    global $_post;
    if (jrCore_checktype($title, 'number_nz') || jrCore_checktype($subtitle, 'number_nz')) {
        $_lang = jrUser_load_lang_strings();
        if (isset($_lang["{$_post['module']}"][$title])) {
            $title = $_lang["{$_post['module']}"][$title];
        }
        if (isset($_lang["{$_post['module']}"][$subtitle])) {
            $subtitle = $_lang["{$_post['module']}"][$subtitle];
        }
    }
    // If this is a Master Admin, they can customize the form they are viewing
    // If it has been registered as a Form Designer form
    if (jrUser_is_master()) {
        if ($_tmp = jrCore_get_registered_module_features('jrCore', 'designer_form')) {
            if (!empty($_post['option']) && isset($_tmp["{$_post['module']}"]["{$_post['option']}"])) {
                $subtitle .= jrCore_page_button('form-designer-button', 'form designer', "jrCore_window_location('" . jrCore_get_base_url() . "/{$_post['module_url']}/form_designer/m={$_post['module']}/v={$_post['option']}')");
            }
        }
    }
    $_tmp = array(
        'type'     => 'page_banner',
        'title'    => $title,
        'subtitle' => $subtitle,
        'icon_url' => $icon,
        'module'   => 'jrCore',
        'template' => 'page_banner.tpl'
    );
    jrCore_create_page_element('page', $_tmp);

    // Set our page title too
    jrCore_page_title($title);
    return true;
}

/**
 * page divider/section
 * @param string $title Title of section
 * @param string $dom_id optional DOM ID for table row
 * @return bool
 */
function jrCore_page_section_header($title, $dom_id = '')
{
    $_tmp = array(
        'type'     => 'page_section_header',
        'title'    => $title,
        'module'   => 'jrCore',
        'template' => 'page_section_header.tpl',
        'id'       => $dom_id
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_notice - show a success/warning/error notice on a page
 *
 * @param string $notice_type Notice Type (success/warning/notice/error)
 * @param string $notice_text Notice Text
 * @param bool $clean_output If true, Notice Text is run through htmlspecialchars()
 * @return bool
 */
function jrCore_page_notice($notice_type, $notice_text, $clean_output = true)
{
    $_lang = jrUser_load_lang_strings();
    if ($clean_output) {
        $notice_text = nl2br(htmlspecialchars($notice_text));
    }
    // Get our lang string
    switch ($notice_type) {
        case 'notice':
            $string = (isset($_lang['jrCore'][22])) ? $_lang['jrCore'][22] : 'notice';
            break;
        case 'warning':
            $string = (isset($_lang['jrCore'][23])) ? $_lang['jrCore'][23] : 'warning';
            break;
        case 'error':
            $string = (isset($_lang['jrCore'][24])) ? $_lang['jrCore'][24] : 'error';
            break;
        case 'success':
            $string = (isset($_lang['jrCore'][25])) ? $_lang['jrCore'][25] : 'success';
            break;
        default:
            $string = $notice_type;
            break;
    }
    $_tmp = array(
        'type'         => 'page_notice',
        'notice_type'  => $notice_type,
        'notice_label' => $string,
        'notice_text'  => $notice_text,
        'module'       => 'jrCore',
        'template'     => 'page_notice.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_link_cell
 * Creates an entry in a form with label and URL in body
 *
 * @param string $label Text for label title
 * @param string $url URL to link to
 * @param string $sublabel Sub label for label
 * @return bool
 */
function jrCore_page_link_cell($label, $url, $sublabel = null)
{
    global $_post;
    $mod = $_post['module'];
    $_ln = jrUser_load_lang_strings();
    if (jrCore_checktype($label, 'number_nz') && isset($_ln[$mod][$label])) {
        $label = $_ln[$mod][$label];
    }
    if (jrCore_checktype($sublabel, 'number_nz') && isset($_ln[$mod][$sublabel])) {
        $sublabel = $_ln[$mod][$sublabel];
    }
    $_tmp = array(
        'type'     => 'page_link_cell',
        'label'    => $label,
        'name'     => jrCore_url_string($label),
        'sublabel' => (is_null($sublabel)) ? false : $sublabel,
        'url'      => $url,
        'module'   => 'jrCore',
        'template' => 'page_link_cell.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_custom - embed html into a page
 * @param string $html Text to embed into page
 * @param string $label Label for Custom HTML
 * @param string $sublabel Label for Custom HTML
 * @param string $help Help drop down for Custom HTML
 * @param string $id the row id so it can be targeted
 * @return bool
 */
function jrCore_page_custom($html, $label = null, $sublabel = null, $help = null, $id = null)
{
    global $_post;
    $_ln = jrUser_load_lang_strings();
    $_tm = array(
        'type'     => 'page_custom',
        'name'     => substr(md5(microtime() . mt_rand()), 8, 8),
        'html'     => $html,
        'label'    => (isset($_ln["{$_post['module']}"][$label])) ? $_ln["{$_post['module']}"][$label] : $label,
        'sublabel' => (isset($_ln["{$_post['module']}"][$sublabel])) ? $_ln["{$_post['module']}"][$sublabel] : $sublabel,
        'help'     => (isset($_ln["{$_post['module']}"][$help])) ? $_ln["{$_post['module']}"][$help] : $help,
        'module'   => 'jrCore',
        'template' => 'page_custom.tpl',
        'id'       => $id
    );
    return jrCore_create_page_element('page', $_tm);
}

/**
 * embeds RAW HTML into the page (no enclosure)
 * @param string $html Text to embed into page
 * @return bool
 */
function jrCore_page_html($html)
{
    $_tmp = array(
        'type'   => 'page_html',
        'html'   => $html,
        'module' => 'jrCore'
    );
    // no template needed for this
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_divider - create a section divider on a page
 * @param null $dom_id
 * @return bool
 */
function jrCore_page_divider($dom_id = null)
{
    $_tmp = array(
        'type'     => 'page_divider',
        'module'   => 'jrCore',
        'dom_id'   => $dom_id,
        'template' => 'page_divider.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Add a vertical space to a page
 * @param null $dom_id
 * @return bool
 */
function jrCore_page_spacer($dom_id = null)
{
    $_tmp = array(
        'type'     => 'page_spacer',
        'module'   => 'jrCore',
        'dom_id'   => $dom_id,
        'template' => 'page_spacer.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Add an anchor to a page
 * @param null $dom_id
 * @return bool
 */
function jrCore_page_anchor($dom_id)
{
    $_tmp = array(
        'type'     => 'page_anchor',
        'module'   => 'jrCore',
        'dom_id'   => $dom_id,
        'template' => 'page_anchor.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Show a page notice
 * @param string $html HTML to show
 * @param string $class CSS Class for background
 * @return bool
 */
function jrCore_page_note($html, $class = 'notice')
{
    $_tmp = array(
        'type'     => 'page_note',
        'html'     => $html,
        'class'    => $class,
        'module'   => 'jrCore',
        'template' => 'page_note.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_template
 * @param string $template Template to embed in page (located in Module/templates)
 * @return bool
 */
function jrCore_page_template($template)
{
    $_tmp = array(
        'type'     => 'page_template',
        'file'     => $template,
        'module'   => 'jrCore',
        'template' => 'page_template.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Create a new tab bar
 * @param array $_tabs Array of tabs to create on page
 * @return bool
 */
function jrCore_page_tab_bar($_tabs, $dom_id = null)
{
    if (isset($_tabs) && is_array($_tabs) && count($_tabs) > 0) {
        $tab_n = count($_tabs);
        $width = round(100 / $tab_n);
        $i     = 1;
        foreach ($_tabs as $k => $_cell) {
            $_tabs[$k]['id']    = 't' . jrCore_url_string($k);
            $_tabs[$k]['width'] = $width;
            $add                = '';
            if (isset($_cell['class'])) {
                $add = " {$_cell['class']}";
            }
            $_tabs[$k]['class'] = 'page_tab';
            // Check for positioning
            if ($i == 1) {
                $_tabs[$k]['class'] .= ' page_tab_first';
            }
            elseif ($i == $tab_n) {
                $_tabs[$k]['class'] .= ' page_tab_last';
            }
            $_tabs[$k]['class'] .= $add;
            $i++;
        }
    }
    $_tmp = array(
        'type'     => 'page_tab_bar',
        'dom_id'   => $dom_id,
        'tabs'     => $_tabs,
        'module'   => 'jrCore',
        'template' => 'page_tab_bar.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Pass in an array of "search areas" that will be prepended to the search string - i.e. "user_name:brian"
 * @param array $_areas Array of areas keys => display values
 * @return bool
 */
function jrCore_page_search_enable_search_areas($_areas)
{
    return jrCore_set_flag('jrcore_page_search_areas', $_areas);
}

/**
 * jrCore_page_search
 * @param string $label Element Label
 * @param string $action Action URL for search form
 * @param string $value default value for search field
 * @param bool $show_help Show Help button true/false
 * @param string $placeholder show place holder
 * @param string $reset_text text for "reset" button
 * @param bool $autofocus add the HTML5 'autofocus' attribute
 * @return bool
 */
function jrCore_page_search($label, $action, $value = null, $show_help = true, $placeholder = '', $reset_text = '', $autofocus = true)
{
    global $_post;
    $show = ' style="display:none"';
    if (isset($_post['search_string'])) {
        $show = '';
    }
    $_lng = jrUser_load_lang_strings();
    $sbtn = $_lng['jrCore'][8];
    if (empty($reset_text)) {
        $reset_text = $_lng['jrCore'][29];
    }
    if (is_null($value) && !empty($_post['search_string'])) {
        $value = $_post['search_string'];
    }
    if (!is_null($value)) {
        $value = str_replace('___AMP', '&', $value);
        $value = str_replace('___FS', '/', $value);
        $value = str_replace('___EQ', '=', $value);
    }
    $action    = jrCore_strip_url_params($action, array('search_string', 'p'));
    $autofocus = ($autofocus) ? ' autofocus="autofocus"' : '';
    $button_id = 'b' . jrCore_create_unique_string(5);

    if ($_areas = jrCore_get_flag('jrcore_page_search_areas')) {
        $jscript = "$('#{$button_id}').jrCore_disable_button();var a=$('#ssta').val();var s=$('#sstr').val().replace('&','___AMP').replace(/\//g,'___FS').replace('=','___EQ');jrCore_window_location('{$action}/search_area='+ jrE(a) +'/search_string='+ jrE(s));return false";
        $html    = '<select id="ssta" class="form_select form_select_area_search" onchange="if ($(\'#sstr\').val().length > 0) { ' . $jscript . ' }">';
        foreach ($_areas as $area => $text) {
            if (!empty($_post['search_area']) && $_post['search_area'] == $area) {
                $html .= '<option value="' . $area . '" selected="selected">' . $text . '</option>';
            }
            else {
                $html .= '<option value="' . $area . '">' . $text . '</option>';
            }
        }
        $html .= '</select><input type="text" name="search_string" id="sstr" placeholder="' . jrCore_entity_string($placeholder) . '"' . $autofocus . ' class="form_text form_text_search form_text_area_search" value="' . jrCore_entity_string($value) . '" onkeydown="if (event && event.keyCode == 13 && this.value.length > 0) {' . $jscript . '}"><input id="' . $button_id . '" type="button" value="' . jrCore_str_to_lower($sbtn) . '" class="form_button form_search_button" onclick="' . $jscript . '"><input type="button" value="' . jrCore_str_to_lower($reset_text) . '" class="form_button form_search_button"' . $show . ' onclick="jrCore_window_location(\'' . $action . '\')">';
        jrCore_delete_flag('jrcore_page_search_areas');
    }
    else {
        $jscript = "$('#{$button_id}').jrCore_disable_button();var s=$('#sstr').val().replace('&','___AMP').replace(/\//g,'___FS').replace('=','___EQ');jrCore_window_location('{$action}/search_string='+ jrE(s));return false";
        $html    = '<input type="text" name="search_string" id="sstr" placeholder="' . jrCore_entity_string($placeholder) . '"' . $autofocus . ' class="form_text form_text_search" value="' . jrCore_entity_string($value) . '" onkeydown="if (event && event.keyCode == 13 && this.value.length > 0) {' . $jscript . '}"><input id="' . $button_id . '" type="button" value="' . jrCore_str_to_lower($sbtn) . '" class="form_button form_search_button" onclick="' . $jscript . '"><input type="button" value="' . jrCore_str_to_lower($reset_text) . '" class="form_button form_search_button"' . $show . ' onclick="jrCore_window_location(\'' . $action . '\')">';
    }
    $_tmp = array(
        'type'      => 'page_search',
        'html'      => $html,
        'label'     => $label,
        'action'    => $action,
        'show_help' => $show_help,
        'value'     => (empty($value)) ? false : $value,
        'module'    => 'jrCore',
        'template'  => 'page_search.tpl',
        '_areas'    => jrCore_get_flag('jrcore_page_search_areas')
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_tool_entry
 * @param string $url Tool URL
 * @param string $label Page element Label
 * @param string $description Page element Description
 * @param string $onclick Javascript onclick content
 * @param string $target Browser anchor target
 * @return bool
 */
function jrCore_page_tool_entry($url, $label, $description, $onclick = null, $target = '_self')
{
    $_tmp = array(
        'type'        => 'page_tool_entry',
        'label'       => $label,
        'label_url'   => $url,
        'description' => $description,
        'onclick'     => (is_null($onclick) || $onclick === false) ? false : $onclick,
        'target'      => $target,
        'module'      => 'jrCore',
        'template'    => 'page_tool_entry.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_table_header
 * @param array $_cells Array containing header row cells
 * @param string $class CSS Class for row
 * @param bool $inline set to TRUE if doing a "header" row inside an open table
 * @param array $_args Additional $_args array passed to event listeners
 * @param string $dom_id option DOM ID for table header
 * @param bool $use_mobile_table Set to FALSE to show desktop table to mobile
 * @param bool $use_template set to FALSE to not use a template for HTML
 * @return bool
 */
function jrCore_page_table_header($_cells, $class = null, $inline = false, $_args = null, $dom_id = null, $use_mobile_table = true, $use_template = null)
{
    if (!$inline) {
        jrCore_delete_flag('jr_html_page_table_row_num');
        jrCore_delete_flag('jr_html_page_table_header_colspan');
    }

    // Let other modules work with our rows
    $_cells = jrCore_trigger_event('jrCore', 'page_table_header', $_cells, $_args);

    $cls = '';
    if (!is_null($class) && strlen($class) > 0) {
        $cls = " {$class}";
    }
    if (is_null($dom_id)) {
        $dom_id = jrCore_create_unique_string(6);
    }
    if (!jrCore_get_flag('jr_html_page_table_header_colspan')) {
        $uniq = count($_cells);
        jrCore_set_flag('jr_html_page_table_header_colspan', $uniq);
    }
    jrCore_set_flag('jrcore_page_table_header_mobile', intval($use_mobile_table));
    jrCore_set_flag('jrcore_page_table_header_cells', $_cells);

    if (is_null($use_template)) {
        $use_template = jrCore_get_config_value('jrCore', 'templated_tables', true);
    }
    if ($use_template) {
        $_tmp = array(
            'type'     => 'page_table_header',
            'cells'    => $_cells,
            'unique'   => $dom_id,
            'class'    => $cls,
            'module'   => 'jrCore',
            'inline'   => $inline,
            'mobile'   => $use_mobile_table,
            'template' => 'page_table_header.tpl'
        );
        return jrCore_create_page_element('page', $_tmp);
    }

    // Fall through - PHP instead of template (for speed)
    $html = '';
    if ($use_mobile_table && jrCore_is_mobile_device() && !jrCore_is_tablet_device()) {
        $html .= "<tr><td colspan=\"2\"><table id=\"page_table_{$dom_id}\" class=\"page_table{$cls}\"><tbody>";
    }
    else {
        if (!$inline) {
            $html .= "<tr><td colspan=\"2\"><table id=\"page_table_{$dom_id}\" class=\"page_table{$cls}\">";
        }
        if (is_array($_cells)) {
            $html .= '<thead><tr class="nodrag nodrop">';
            foreach ($_cells as $_cell) {
                $width = (!empty($_cell['width'])) ? " style=\"width:{$_cell['width']}\"" : '';
                $class = (isset($_cell['class'])) ? 'page_table_header ' . $_cell['class'] : 'page_table_header';
                $html  .= "<th class=\"{$class}\"{$width}>{$_cell['title']}</th>";
            }
            $html .= '</tr></thead>';
        }
        $html .= '<tbody>';
    }
    $_tmp = array(
        'type'   => 'page_table_header',
        'module' => 'jrCore',
        'html'   => $html
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_table_row
 * @param array $_cells Array containing row cells
 * @param string $class CSS Class for row
 * @param array $_args Additional $_args array passed to event listeners
 * @param bool $use_template set to FALSE to not use a template for HTML
 * @return bool
 */
function jrCore_page_table_row($_cells, $class = null, $_args = null, $increment_row_num = true, $use_template = null)
{
    $rownum = jrCore_get_flag('jr_html_page_table_row_num');
    if (!$rownum) {
        $rownum = 0;
    }
    $colspan = jrCore_get_flag('jr_html_page_table_header_colspan');
    $col_cnt = count($_cells);
    ksort($_cells, SORT_NUMERIC);
    if ($colspan > $col_cnt) {
        // Adjust our last row in our cells to span the entire width
        $_tmp            = array_pop($_cells);
        $_tmp['colspan'] = ' colspan="' . $colspan . '"';
        if ($col_cnt == 1) {
            $_cells = array($_tmp);
        }
        else {
            $_cells[] = $_tmp;
        }
    }
    foreach ($_cells as $k => $v) {
        if (!isset($v['colspan'])) {
            $_cells[$k]['colspan'] = '';
        }
    }

    // Let other modules work with our rows
    $_cells = jrCore_trigger_event('jrCore', 'page_table_row', $_cells, $_args);

    $cls = '';
    if (!is_null($class) && strlen($class) > 0) {
        $cls = " {$class}";
    }

    // If we are on a mobile device we are going to render the table
    // differently with each row it's own small section - we need the
    // header titles for each row
    if ($_th = jrCore_get_flag('jrcore_page_table_header_cells')) {
        foreach ($_th as $k => $v) {
            if (isset($_cells[$k])) {
                $_cells[$k]['header_title'] = $v['title'];
            }
        }
    }

    $use_mobile_table = (bool) jrCore_get_flag('jrcore_page_table_header_mobile');
    if (is_null($use_template)) {
        $use_template = jrCore_get_config_value('jrCore', 'templated_tables', true);
    }
    if ($use_template) {

        $_tmp = array(
            'type'     => 'page_table_row',
            'cells'    => $_cells,
            'cellnum'  => $col_cnt,
            'class'    => $cls,
            'module'   => 'jrCore',
            'mobile'   => $use_mobile_table,
            'template' => 'page_table_row.tpl'
        );
        if ($increment_row_num) {
            $_tmp['rownum'] = ++$rownum;
        }
        jrCore_set_flag('jr_html_page_table_row_num', $rownum);
        return jrCore_create_page_element('page', $_tmp);
    }

    $row_class = "page_table_row_alt{$cls}";
    if ($rownum % 2 === 0) {
        $row_class = "page_table_row{$cls}";
    }

    $html = '';
    if ($use_mobile_table && jrCore_is_mobile_device() && !jrCore_is_tablet_device()) {
        $idx = 1;
        foreach ($_cells as $_cell) {
            $ccls = (!empty($_cell['class'])) ? " {$_cell['class']}" : '';
            $html .= "<tr class=\"{$row_class}\" data-row-num=\"{$idx}\"><td class=\"page_table_header page_table_cell_mobile_left\">{$_cell['header_title']}</td><td class=\"page_table_cell page_table_cell_mobile_right{$ccls}\">{$_cell['title']}</td></tr>";
        }
    }
    else {
        $html .= "<tr class=\"{$row_class}\">";
        foreach ($_cells as $_cell) {
            $wdth = (!empty($_cell['width'])) ? " style=\"width:{$_cell['width']}\"" : '';
            $ccls = (!empty($_cell['class'])) ? " {$_cell['class']}" : '';
            $html .= "<td class=\"page_table_cell{$ccls}\"{$wdth}{$_cell['colspan']}>{$_cell['title']}</td>";
        }
        $html .= '</tr>';
    }
    $_tmp = array(
        'type'   => 'page_table_row',
        'module' => 'jrCore',
        'html'   => $html
    );
    if ($increment_row_num) {
        $_tmp['rownum'] = ++$rownum;
    }
    jrCore_set_flag('jr_html_page_table_row_num', $rownum);
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Get the number of rows to use for pagebreak
 * @return int
 */
function jrCore_get_pager_rows()
{
    if ($rows = jrCore_get_cookie('rows')) {
        return intval($rows);
    }
    // jrcore_pager_rows is deprecated - use 'rows' cookie
    elseif (!empty($_COOKIE['jrcore_pager_rows'])) {
        return intval($_COOKIE['jrcore_pager_rows']);
    }
    return 12;
}

/**
 * Create pager for page table
 * @param array $_page page array page elements for pager including:
 *        'prev_page_num' => previous page number
 *        'this_page_num' => current page number
 *        'next_page_num' => next page number
 *        'total_pages'   => total number of pages without LIMIT clause
 * @param array $_xtra Array containing information about current data set (as returned from dbPagedQuery())
 * @param bool $show_jumper Set to FALSE to prevent jumper from showing
 * @return bool
 */
function jrCore_page_table_pager($_page, $_xtra = null)
{
    // We have to strip the page number (p) as well as any
    // other _xtra args we get so we don't duplicate them
    if (is_array($_xtra)) {
        $_strip      = $_xtra;
        $_strip['p'] = 1;
    }
    else {
        $_strip = array('p' => 1);
    }
    $this_page_url = jrCore_strip_url_params(jrCore_get_current_url(), array_keys($_strip));
    if (!empty($_page['info']['this_page_url'])) {
        $this_page_url = $_page['info']['this_page_url'];
    }
    $this_page_url = str_replace('%', '%25', $this_page_url);
    $this_page_url = str_replace(' ', '%20', $this_page_url);

    // If we have $_xtra, it means we need to add additional url vars
    if (is_array($_xtra)) {
        foreach ($_xtra as $k => $v) {
            // if we have a search string in the page next/prev replace it
            if ($k == 'search_string') {
                $v = str_replace('&', '___AMP', $v);
                $v = str_replace('/', '___FS', $v);
                $v = str_replace('=', '___EQ', $v);
            }
            $this_page_url .= "/{$k}=" . urlencode($v);
        }
    }
    // We always show the pager
    if (is_array($_page) && isset($_page['info']) && (isset($_page['info']['simplepagebreak']) || isset($_page['info']['total_pages']))) {

        $prev_page_url = '';
        if (!empty($_page['info']['prev_page_url'])) {
            $prev_page_url = $_page['info']['prev_page_url'];
        }
        elseif (!empty($_page['info']['prev_page']) && jrCore_checktype($_page['info']['prev_page'], 'number_nz')) {
            $prev_page_url = "{$this_page_url}/p={$_page['info']['prev_page']}";
        }

        $next_page_url = '';
        if (!empty($_page['info']['next_page_url'])) {
            $next_page_url = $_page['info']['next_page_url'];
        }
        elseif (!empty($_page['info']['next_page']) && jrCore_checktype($_page['info']['next_page'], 'number_nz')) {
            $next_page_url = "{$this_page_url}/p={$_page['info']['next_page']}";
        }

        if (isset($_page['info']['total_pages']) && $_page['info']['total_pages'] > 0) {

            $end = ($_page['info']['total_pages'] - 100);
            $mdl = 0;
            if ($_page['info']['this_page'] > 50) {
                $mdl = ($_page['info']['this_page'] - 50);
            }
            $mdh      = ($_page['info']['this_page'] + 50);
            $i        = 1;
            $_options = array();
            while ($i <= $_page['info']['total_pages']) {
                if ($i < 100 || $i > $end || ($i > $mdl && $i < $mdh)) {
                    $_options[$i] = $i;
                }
                $i++;
            }

            $onchange    = (jrCore_checktype($this_page_url, 'url')) ? 'jrCore_window_location(\'' . $this_page_url . '/p=\'+ $(this).val())' : $this_page_url;
            $page_jumper = jrCore_page_jumper('p', $_options, $_page['info']['this_page'], $onchange, 'page-table-jumper-page');
        }
        else {
            $page_jumper = $_page['info']['this_page'];
        }

        if (!isset($_page['disable_page_select']) || jrCore_checktype($_page['disable_page_select'], 'is_false')) {
            // Allow the user to override the default number of items per page
            $pagebreak = jrCore_get_pager_rows();
            $_options  = array();
            $_numbers  = (!empty($_xtra['_perpage']) && is_array($_xtra['_perpage'])) ? $_xtra['_perpage'] : array(5, 8, 10, 12, 15, 20, 25, 30, 40, 50, 60, 75, 100);
            foreach ($_numbers as $per_page) {
                $_options[$per_page] = $per_page;
            }
            $onchange    = (jrCore_checktype($this_page_url, 'url')) ? 'var r=$(this).val(); jrCore_set_pager_rows(r, function() { jrCore_window_location(\'' . $this_page_url . '\'); })' : $this_page_url;
            $page_select = jrCore_page_jumper('r', $_options, $pagebreak, $onchange, 'page-table-jumper-perpage');
        }
        else {
            $page_select = false;
        }

        $_tmp = array(
            'type'          => 'page_table_pager',
            'prev_page_url' => $prev_page_url,
            'this_page_url' => $this_page_url,
            'next_page_url' => $next_page_url,
            'prev_page_num' => (isset($_page['info']['prev_page'])) ? $_page['info']['prev_page'] : 0,
            'this_page_num' => $_page['info']['this_page'],
            'next_page_num' => $_page['info']['next_page'],
            'total_pages'   => (isset($_page['info']['total_pages'])) ? jrCore_number_format($_page['info']['total_pages']) : '',
            'page_jumper'   => $page_jumper,
            'page_select'   => $page_select,
            'colspan'       => jrCore_get_flag('jr_html_page_table_header_colspan'),
            'mobile'        => (bool) jrCore_get_flag('jrcore_page_table_header_mobile'),
            'module'        => 'jrCore',
            'template'      => 'page_table_pager.tpl'
        );
        jrCore_create_page_element('page', $_tmp);
    }
    jrCore_delete_flag('jr_html_page_table_header_colspan');
    return true;
}

/**
 * Create a footer row for a Page Table
 * @param null $_cells array Cells in footer
 * @param null $class string Row Class
 * @param array $_args Additional $_args array passed to event listeners
 * @param bool $use_template set to FALSE to not use a template for HTML
 * @return bool
 */
function jrCore_page_table_footer($_cells = null, $class = null, $_args = null, $use_template = null)
{
    // Let other modules work with our rows
    $_cells = jrCore_trigger_event('jrCore', 'page_table_footer', $_cells, $_args);

    $cls = '';
    if (!is_null($class) && strlen($class) > 0) {
        $cls = " {$class}";
    }

    if (is_null($use_template)) {
        $use_template = jrCore_get_config_value('jrCore', 'templated_tables', true);
    }
    if ($use_template) {

        $_tmp = array(
            'type'     => 'page_table_footer',
            'cells'    => $_cells,
            'class'    => $cls,
            'module'   => 'jrCore',
            'template' => 'page_table_footer.tpl'
        );
        $uniq = jrCore_get_flag('jr_html_page_table_footer_colspan');
        if (!$uniq && is_array($_cells)) {
            $uniq = count($_cells);
            jrCore_set_flag('jr_html_page_table_footer_colspan', $uniq);
        }
        jrCore_delete_flag('jrcore_page_table_header_mobile');
        return jrCore_create_page_element('page', $_tmp);
    }

    $html = '';
    if (is_array($_cells)) {
        $html .= '<tr class="nodrag nodrop">';
        foreach ($_cells as $_cell) {
            $wdth = (!empty($_cell['width'])) ? " style=\"width:{$_cell['width']}\"" : '';
            $ccls = (!empty($_cell['class'])) ? " {$_cell['class']}" : '';
            $html .= "<th class=\"page_table_footer{$ccls}\"{$wdth}>{$_cell['title']}</th>";
        }
        $html .= '</tr>';
    }
    $html .= '</tbody></table></td></tr>';
    $_tmp = array(
        'type'   => 'page_table_row',
        'module' => 'jrCore',
        'html'   => $html
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * jrCore_page_cancel_button
 * @param string $cancel_url URL to redirect browser to when cancel button is clicked
 * @param string $cancel_text button text value for cancel button
 * @return bool
 */
function jrCore_page_cancel_button($cancel_url, $cancel_text = null)
{
    global $_post;
    switch ($cancel_url) {
        case 'referrer':
            $cancel_url = "history.back();";
            break;
        default:
            if (strpos(' ' . $cancel_url, 'modal') && strpos(' ' . $cancel_url, 'close')) {
                $cancel_url = '$.modal.close();';
            }
            elseif (jrCore_checktype($cancel_url, 'url')) {
                $cancel_url = "jrCore_window_location('{$cancel_url}')";
            }
            break;
    }
    if (is_null($cancel_text) || $cancel_text === false) {
        $_lang       = jrUser_load_lang_strings();
        $cancel_text = (isset($_lang['jrCore'][2])) ? $_lang['jrCore'][2] : 'cancel';
    }
    elseif (jrCore_checktype($cancel_text, 'number_nz')) {
        $_lang = jrUser_load_lang_strings();
        if (isset($_lang["{$_post['module']}"][$cancel_text])) {
            $cancel_text = $_lang["{$_post['module']}"][$cancel_text];
        }
    }
    $html = '<input type="button" class="form_button" value="' . jrCore_str_to_lower($cancel_text) . '" onclick="' . $cancel_url . '">';
    $_tmp = array(
        'type'     => 'page_cancel_button',
        'html'     => $html,
        'module'   => 'jrCore',
        'template' => 'page_cancel_button.tpl'
    );
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Create a "Close" button that closes a popup window
 * @param string $onclick Onclick function
 * @return bool
 */
function jrCore_page_close_button($onclick = 'self.close();')
{
    $_lng = jrUser_load_lang_strings();
    $cbtn = (isset($_lng['jrCore'][28])) ? $_lng['jrCore'][28] : 'close';
    $html = '<input type="button" class="form_button" value="' . jrCore_str_to_lower($cbtn) . '" onclick="' . $onclick . '">';
    $_tmp = array(
        'type'     => 'page_close_button',
        'html'     => $html,
        'module'   => 'jrCore',
        'template' => 'page_close_button.tpl'
    );
    // template not needed here
    return jrCore_create_page_element('page', $_tmp);
}

/**
 * Do not include header.tpl or footer.tpl in processed output
 * @return bool
 */
function jrCore_page_set_no_header_or_footer()
{
    return jrCore_set_flag('jrcore_page_no_header_or_footer', true);
}

/**
 * Only include meta.tpl in processed output
 * @return bool
 */
function jrCore_page_set_meta_header_only()
{
    return jrCore_set_flag('jrcore_page_meta_header_only', true);
}

/**
 * Include the admin menu in the processed output
 * @return bool
 */
function jrCore_page_include_admin_menu()
{
    return jrCore_set_flag('jrcore_page_include_admin_menu', true);
}

/**
 * Defer processing of JS and CSS
 * @note This is used when capturing the output of jrCore_page_display() for further processing before display
 * @return bool
 */
function jrCore_page_defer_js_and_css()
{
    return jrCore_set_flag('jrcore_page_defer_js_and_css', true);
}

/**
 * Set a custom page class
 * @param string $class
 * @return bool
 */
function jrCore_page_set_custom_page_class($class)
{
    return jrCore_set_flag('jrcore_page_class', $class);
}

/**
 * jrCore_get_module_index
 *
 * @param string $module module string module name
 * @return string
 */
function jrCore_get_module_index($module)
{
    // If our module is NOT active, show info
    if (!jrCore_module_is_active($module)) {
        return 'admin/info';
    }

    // We need to go through each module and get it's default page
    $_df = jrCore_get_registered_module_features('jrCore', 'default_admin_view');

    if (isset($_df[$module]) && is_array($_df[$module])) {
        $_tmp = array_keys($_df[$module]);
        return reset($_tmp);
    }

    if (is_file(APP_DIR . "/modules/{$module}/config.php")) {
        return 'admin/global';
    }

    $_quota = jrCore_get_registered_module_features('jrCore', 'quota_support');
    if (isset($_quota[$module]) || is_file(APP_DIR . "/modules/{$module}/quota.php")) {
        return 'admin/quota';
    }

    // Get registered tool views
    $_tool = jrCore_get_registered_module_features('jrCore', 'tool_view');
    if (isset($_tool[$module])) {
        return 'admin/tools';
    }

    if (isset($_lang[$module])) {
        return 'admin/language';
    }

    // all modules have an info panel
    return 'admin/info';
}

/**
 * Verify pending items are correct
 * @return bool
 */
function jrCore_verify_pending_items()
{
    // First - see if we have any LINKED items that need to be removed or approved
    $tbl = jrCore_db_table_name('jrCore', 'pending');
    $req = "SELECT pending_id AS pid, pending_linked_item_module AS plmod, pending_linked_item_id AS plid FROM {$tbl} WHERE pending_linked_item_id > 0";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        $_md = array();
        foreach ($_rt as $_p) {
            if (!isset($_md["{$_p['plmod']}"])) {
                $_md["{$_p['plmod']}"] = array();
            }
            $iid                         = (int) $_p['plid'];
            $_md["{$_p['plmod']}"][$iid] = $_p['pid'];
        }
        if (count($_md) > 0) {
            $_db = array();
            $num = 0;
            foreach ($_md as $mod => $_ids) {
                if (jrCore_is_datastore_module($mod)) {
                    if ($pfx = jrCore_db_get_prefix($mod)) {
                        $_dl = $_ids;
                        $_tm = jrCore_db_get_multiple_items($mod, array_keys($_ids), array('_item_id', "{$pfx}_pending"));
                        if ($_tm && is_array($_tm)) {
                            foreach ($_tm as $_i) {
                                if (isset($_i["{$pfx}_pending"]) && $_i["{$pfx}_pending"] >= 1) {
                                    // We are still pending...
                                    $iid = (int) $_i['_item_id'];
                                    unset($_dl[$iid]);
                                }
                            }
                        }
                        if (count($_dl) > 0) {
                            // We have items that are no longer pending - approve
                            $req = "SELECT pending_id AS pid, pending_module AS pmod, pending_item_id AS plid FROM {$tbl} WHERE pending_id IN(" . implode(',', $_dl) . ')';
                            $_pi = jrCore_db_query($req, 'NUMERIC');
                            if ($_pi && is_array($_pi)) {
                                $_pm = array();
                                foreach ($_pi as $_ap) {
                                    if (!isset($_pm["{$_ap['pmod']}"])) {
                                        $_pm["{$_ap['pmod']}"] = array();
                                    }
                                    if ($pxx = jrCore_db_get_prefix($_ap['pmod'])) {
                                        $_pm["{$_ap['pmod']}"]["{$_ap['plid']}"] = array("{$pxx}_pending" => 0);
                                    }
                                }
                                if (count($_pm) > 0) {
                                    foreach ($_pm as $pmod => $_up) {
                                        jrCore_db_update_multiple_items($pmod, $_up);
                                    }
                                }
                            }

                            // Cleanup
                            $req = "DELETE FROM {$tbl} WHERE pending_id IN(" . implode(',', $_dl) . ')';
                            $cnt = jrCore_db_query($req, 'COUNT');
                            if ($cnt > 0) {
                                $num       += $cnt;
                                $_db[$mod] = $_dl;
                            }
                        }
                    }
                }
            }
            if ($num > 0) {
                jrCore_logger('INF', 'core: deleted ' . jrCore_number_format($num) . ' invalid pending item entries', $_db);
            }
        }
    }
    return true;
}

/**
 * Get array of modules that have pending items
 * @return array|false
 */
function jrCore_get_dashboard_pending_modules()
{
    $key = 'jr_pending_modules';
    if (!$_pi = jrCore_get_flag($key)) {
        $tbl = jrCore_db_table_name('jrCore', 'pending');
        $req = "SELECT DISTINCT(pending_module) AS m FROM {$tbl} WHERE pending_linked_item_id = 0";
        $_pi = jrCore_db_query($req, 'm', false, 'm');
        if (!is_array($_pi)) {
            $_pi = 'no_items';
        }
        jrCore_set_flag($key, $_pi);
    }
    return (is_array($_pi)) ? $_pi : false;
}

/**
 * Get the default dashboard pending tab
 * @return mixed|string
 */
function jrCore_get_dashboard_default_pending_tab()
{
    // Do we have pending items?
    $mod = 'jrUser';
    if ($_pi = jrCore_get_dashboard_pending_modules()) {
        if (!isset($_pi['jrUser'])) {
            $mod = reset($_pi);
        }
    }
    return $mod;
}

/**
 * Tabs for Dashboard
 * @param string $active active tab
 * @return bool
 */
function jrCore_page_dashboard_tabs($active = 'online')
{
    // Our Tabs for the top of the dashboard view
    $_tabs = array();

    $curl = jrCore_get_module_url('jrCore');
    if (jrUser_is_master()) {
        $_tabs['acp'] = array(
            'label' => 'ACP',
            'url'   => jrCore_get_base_url() . "/{$curl}/admin/global"
        );
    }
    $_tabs['bigview']      = array(
        'label' => 'dashboard',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/bigview"
    );
    $_tabs['online']       = array(
        'label' => 'users online',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/online"
    );
    $_tabs['pending']      = array(
        'label' => 'pending',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/pending",
    );
    $_tabs['activity']     = array(
        'label' => 'activity log',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/activity"
    );
    $purl                  = jrCore_get_module_url('jrUser');
    $_tabs['browser']      = array(
        'label' => 'data browser',
        'url'   => jrCore_get_base_url() . "/{$purl}/dashboard/browser"
    );
    $_tabs['recycle_bin']  = array(
        'label' => 'recycle bin',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/recycle_bin"
    );
    $_tabs['queue_viewer'] = array(
        'label' => 'queue viewer',
        'url'   => jrCore_get_base_url() . "/{$curl}/dashboard/queue_viewer"
    );

    // Let other modules integrate with dashboard
    $_tabs = jrCore_trigger_event('jrCore', 'dashboard_tabs', $_tabs);

    if (!empty($_tabs[$active])) {
        $_tabs[$active]['active'] = true;
    }
    jrCore_set_flag('jrcore_dashboard_active', 1);
    jrCore_page_tab_bar($_tabs);
    return true;
}

/**
 * Return TRUE of a module has a visible config (non-hidden)
 * @param $module string Module
 * @return bool
 */
function jrCore_module_has_visible_config($module)
{
    if (!$_tm = jrCore_get_flag('jrCore_module_has_visible_config')) {
        $tbl = jrCore_db_table_name('jrCore', 'setting');
        $req = "SELECT `module` AS m FROM {$tbl} WHERE `type` != 'hidden'";
        $_tm = jrCore_db_query($req, 'm', false, 'm');
        jrCore_set_flag('jrCore_module_has_visible_config', $_tm);
    }
    if (isset($_tm[$module])) {
        return true;
    }
    return false;
}

/**
 * Module Tabs in ACP
 * @param string $module module string module name
 * @param string $active active string active tab can be one of: global,quota,tools,language,templates,info
 */
function jrCore_page_admin_tabs($module, $active = 'tools')
{
    $_lang = jrUser_load_lang_strings();

    // Get registered tool views
    $_tools = jrCore_get_registered_module_features('jrCore', 'tool_view');
    $_quota = jrCore_get_registered_module_features('jrCore', 'quota_support');

    // Our current module url
    $url = jrCore_get_module_url($module);

    // Our admin tabs for the top of the view
    $_tabs = array();
    if (jrCore_module_is_active($module) && jrCore_module_has_visible_config($module)) {
        $_tabs['global'] = array(
            'label' => 'global config',
            'url'   => jrCore_get_base_url() . "/{$url}/admin/global"
        );
    }
    if (jrCore_module_is_active($module) && (isset($_quota[$module]) || is_file(APP_DIR . "/modules/{$module}/quota.php"))) {
        if (isset($_quota[$module]) || strpos(jrCore_file_get_contents(APP_DIR . "/modules/{$module}/quota.php"), 'jrProfile_register_quota_setting')) {
            $_tabs['quota'] = array(
                'label' => 'quota config',
                'url'   => jrCore_get_base_url() . "/{$url}/admin/quota"
            );
        }
    }

    // Check for additional tabs registered by the module
    $_tmp = jrCore_get_registered_module_features('jrCore', 'admin_tab');
    $_tmp = (isset($_tmp[$module])) ? $_tmp[$module] : false;
    if (is_array($_tmp)) {
        $_tab = array();
        $murl = jrCore_get_module_url($module);
        foreach ($_tmp as $view => $label) {
            // There are some views we cannot set
            switch ($view) {
                case 'global':
                case 'quota':
                case 'tools':
                case 'language':
                case 'templates':
                case 'style':
                case 'images':
                case 'info':
                    continue 2;
            }
            $_tab[$view] = array(
                'label' => $label,
                'url'   => jrCore_get_base_url() . "/{$murl}/{$view}"
            );
        }
        $_tabs = $_tabs + $_tab;
    }

    if (jrCore_module_is_active($module) && (isset($_tools[$module]) || jrCore_db_get_prefix($module))) {
        $_tabs['tools'] = array(
            'label' => 'tools',
            'url'   => jrCore_get_base_url() . "/{$url}/admin/tools"
        );
    }
    else {
        // We can't set tools for out default here as there is no tools...
        if ($active == 'tools') {
            $active = 'info';
        }
    }
    if (jrCore_module_is_active($module) && isset($_lang[$module])) {
        $_tabs['language'] = array(
            'label' => 'language',
            'url'   => jrCore_get_base_url() . "/{$url}/admin/language"
        );
    }
    if (jrCore_module_is_active($module) && is_dir(APP_DIR . "/modules/{$module}/img")) {
        $_pt = array('png', 'jpg', 'gif');
        $_fl = array();
        foreach ($_pt as $pattern) {
            $_fl = array_merge($_fl, glob(APP_DIR . "/modules/{$module}/img/*.{$pattern}"));
        }
        if (count($_fl) > 0) {
            $_tabs['images'] = array(
                'label' => 'images',
                'url'   => jrCore_get_base_url() . "/{$url}/admin/images"
            );
        }
    }
    if (jrCore_module_is_active($module) && is_dir(APP_DIR . "/modules/{$module}/templates")) {
        $_tabs['templates'] = array(
            'label' => 'templates',
            'url'   => jrCore_get_base_url() . "/{$url}/admin/templates"
        );
    }
    $_tabs['info'] = array(
        'label' => 'info',
        'url'   => jrCore_get_base_url() . "/{$url}/admin/info"
    );

    if (isset($_tabs[$active])) {
        $_tabs[$active]['active'] = true;
    }
    // admin_tabs event
    $_args = array(
        'active' => $active
    );
    $_tabs = jrCore_trigger_event('jrCore', 'admin_tabs', $_tabs, $_args);
    jrCore_page_tab_bar($_tabs);
}

/**
 * Create tabs for each module with pending items
 * @param string $active
 * @return bool
 */
function jrCore_dashboard_pending_tabs($active = 'jrUser')
{
    global $_mods, $_post;
    if (!$_rt = jrCore_get_dashboard_pending_modules()) {
        $_rt = array();
    }
    // Always show the module that comes in
    if (isset($_post['m']) && jrCore_module_is_active($_post['m']) && !isset($_rt["{$_post['m']}"])) {
        $_rt["{$_post['m']}"] = array('pending_module' => $_post['m']);
    }
    // Always show Users
    if (!isset($_rt['jrUser'])) {
        $_rt['jrUser'] = array('pending_module' => 'jrUser');
    }
    $_tabs = array();
    foreach ($_rt as $m => $v) {
        if (jrCore_module_is_active($m)) {
            $_tabs[$m] = array(
                'label' => $_mods[$m]['module_name'],
                'url'   => jrCore_get_base_url() . "/{$_post['module_url']}/dashboard/pending/m={$m}"
            );
        }
    }
    if (isset($_tabs[$active])) {
        $_tabs[$active]['active'] = true;
    }
    if (count($_tabs) > 0) {
        uasort($_tabs, function ($a, $b) {
            return strcasecmp($a['label'], $b['label']);
        });
        jrCore_page_tab_bar($_tabs);
    }
    return true;
}

/**
 * Tabs shown for a skin in the ACP
 * @param string $skin Active Skin
 * @param string $active active string active tab can be one of: global,style,images,language,templates,info
 */
function jrCore_page_skin_tabs($skin, $active = 'info')
{
    // Core Module URL
    $url   = jrCore_get_module_url('jrCore');
    $_tabs = array();
    if (is_file(APP_DIR . "/skins/{$skin}/config.php")) {
        $_tabs['global'] = array(
            'label' => 'global config',
            'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/global/skin={$skin}"
        );
    }
    $_tabs['style']  = array(
        'label' => 'style',
        'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/style/skin={$skin}"
    );
    $_tabs['images'] = array(
        'label' => 'images',
        'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/images/skin={$skin}"
    );
    if (is_dir(APP_DIR . "/skins/{$skin}/lang")) {
        $_tabs['language'] = array(
            'label' => 'language',
            'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/language/skin={$skin}"
        );
    }
    $_tabs['templates'] = array(
        'label' => 'templates',
        'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/templates/skin={$skin}"
    );
    $_tabs['info']      = array(
        'label' => 'info',
        'url'   => jrCore_get_base_url() . "/{$url}/skin_admin/info/skin={$skin}"
    );
    if (isset($_tabs[$active])) {
        $_tabs[$active]['active'] = true;
    }

    // skin_tabs event
    $_args = array(
        'active' => $active
    );
    $_tabs = jrCore_trigger_event('jrCore', 'skin_tabs', $_tabs, $_args);

    jrCore_page_tab_bar($_tabs);
}

/**
 * Add Javascript for ACP accordion menu to page
 * @param $category string default category
 * @return bool
 */
function jrCore_admin_menu_accordion_js($category)
{
    $cat = strtolower($category);
    // We want to hide ALL categories except the category we are currently working in.
    $hide = "\n" . 'var allPanels = $(\'.accordion > dd\').hide();$(\'.accordion > dd[id="c' . $cat . '"]\').show();' . "\n";
    $_js  = array('(function($) { ' . $hide . '
    $(\'.accordion > a > dt\').click(function() {
    var p = $(this).parent().next();
    if (p.is(\':hidden\')) { allPanels.slideUp(); p.slideDown(); }
    return false; }); })(jQuery);');
    return jrCore_create_page_element('javascript_ready_function', $_js);
}

/**
 * Parse a set of page elements and display them
 * @param bool $return_html Set to true to return HTML instead of display
 * @param string $body_id
 * @return bool|string
 */
function jrCore_page_display($return_html = false, $body_id = null)
{
    global $_post, $_mods;
    // See if have an open form on the page - if we do, close it up
    // with our submit and and bring it into the page

    // See if we are doing an ADMIN MENU VIEW for this module/view
    $admn = jrCore_get_flag('jrcore_page_include_admin_menu');
    if ($admn) {
        if (isset($_post['skin'])) {

            $_adm                     = array(
                'active_tab' => 'skins',
                '_skins'     => jrCore_get_acp_skins()
            );
            $_mta                     = jrCore_skin_meta_data($_post['skin']);
            $_adm['default_category'] = 'general';
            if (isset($_mta['category']) && strlen($_mta['category']) > 0) {
                $_adm['default_category'] = trim(strtolower($_mta['category']));
            }
        }
        else {
            $_adm = array(
                '_modules'   => array(),
                'active_tab' => 'modules'
            );

            $_tmp = array();
            $_ina = array();
            foreach ($_mods as $mod_dir => $_inf) {
                if ($_inf['module_active'] == '1') {
                    $_tmp[$mod_dir] = $_inf['module_name'];
                }
                else {
                    $_ina[$mod_dir] = $_inf['module_name'];
                }
            }
            natcasesort($_tmp);
            natcasesort($_ina);
            $_tmp = array_merge($_tmp, $_ina);

            $_out = array();
            foreach ($_tmp as $mod_dir => $ignored) {
                if (!isset($_mods[$mod_dir]['module_category'])) {
                    $_mods[$mod_dir]['module_category'] = 'tools';
                }
                $cat = $_mods[$mod_dir]['module_category'];
                if (!isset($_out[$cat])) {
                    $_out[$cat] = array();
                }
                $_out[$cat][$mod_dir] = $_mods[$mod_dir];
            }
            $_adm['_modules']['core'] = $_out['core'];
            unset($_out['core']);
            $_adm['_modules'] = $_adm['_modules'] + $_out;
            ksort($_adm['_modules']);
            unset($_out);

            $_adm['default_category'] = 'core';
            if (!empty($_mods["{$_post['module']}"]['module_category'])) {
                $_adm['default_category'] = $_mods["{$_post['module']}"]['module_category'];
            }
        }
        jrCore_admin_menu_accordion_js($_adm['default_category']);
    }

    // See if we have an active form session
    $_form = jrCore_form_get_session();

    $design = false;
    // Make sure we have not already displayed this form (i.e. the form is embedded into another page)
    $tmp = false;
    if (!empty($_form['form_token'])) {
        $tmp = jrCore_get_flag("jrcore_page_display_form_{$_form['form_token']}");
    }
    if (!$tmp && is_array($_form) && isset($_form['form_params'])) {

        $module               = $_form['form_params']['module'];
        $_form['form_fields'] = jrCore_get_flag('jrcore_form_session_fields');

        // If our form info changes from a listener, reload
        $_tfrm = jrCore_get_flag('jrcore_form_session_fields');
        if ($_tfrm !== $_form['form_fields']) {
            $_form['form_fields'] = $_tfrm;
        }
        unset($_tfrm);

        // Check and see if this form is registered with the form designer
        if (isset($_form['form_fields']) && is_array($_form['form_fields']) && count($_form['form_fields']) > 0) {

            $_tmp = jrCore_get_registered_module_features('jrCore', 'designer_form');

            // if our install flag is set, and this form has registered for the form designer, we need to make sure we are setup.
            if (isset($_tmp["{$_post['module']}"]["{$_post['option']}"])) {

                $_fld = jrCore_get_designer_form_fields($_post['module'], $_post['option']);
                $_sfd = $_fld;  // backup

                if (jrUser_is_master()) {
                    // This is a designer form - make sure the fields are setup
                    foreach ($_form['form_fields'] as $k => $_field) {
                        if (!$_fld || !is_array($_fld) || !isset($_fld["{$_field['name']}"])) {
                            $_field['active'] = 1;
                            $_field['order']  = ($k + 1);
                            jrCore_verify_designer_form_field($_post['module'], $_post['option'], $_field);
                        }
                    }
                }

                // Next - let's get all our designer info about this module/view so we can override
                // what is coming in from the actual module view
                if (is_array($_fld)) {
                    $design = true;
                    // Go through and remove the fields we have already substituted
                    foreach ($_form['form_fields'] as $_field) {
                        if (isset($_field['name']) && isset($_fld["{$_field['name']}"])) {
                            unset($_fld["{$_field['name']}"]);
                        }
                        elseif (isset($_field['live_search_original_name']) && isset($_fld["{$_field['live_search_original_name']}"])) {
                            unset($_fld["{$_field['live_search_original_name']}"]);
                        }
                    }
                    // See if we have any NEW fields left over
                    if (is_array($_fld) && count($_fld) > 0) {
                        $_val = jrCore_get_flag('jrcore_form_create_values');
                        foreach ($_fld as $_field) {
                            if (isset($_field['active']) && $_field['active'] == '1') {
                                // If this is a meter based field, we need to pass in a copy of the active item id as "value".
                                if (!isset($_field['value'])) {
                                    if (isset($_val["{$_field['name']}_size"])) {
                                        // We have a file based field - add in value
                                        $_field['value'] = $_val;
                                    }
                                }
                                // Make sure image_delete is added in for images
                                if (isset($_field['type']) && $_field['type'] == 'image') {
                                    $_field['image_delete'] = true;
                                }
                                $_field['is_form_designer_field'] = true;
                                jrCore_form_field_create($_field, $module, null, false);
                            }
                        }
                    }
                }
            }

            // If this is the FIRST load of a form that is a designer form, $_fld will be
            // empty and $design will be false - we change that here if needed.
            if (isset($_tmp["{$_post['module']}"]["{$_post['option']}"])) {
                $design = true;
            }

            // Bring in any additional form fields added by modules
            $_form = jrCore_trigger_event('jrCore', 'form_display', $_form);

            // Make sure additional fields form listeners are added in
            $_form['form_fields'] = jrCore_get_flag('jrcore_form_session_fields');
        }

        // Bring in lang strings
        $_lang = jrUser_load_lang_strings();

        $undo = false;
        if (isset($_form['form_params']['reset'])) {
            if (isset($_form['form_params']['reset_value']) && isset($_lang[$module]["{$_form['form_params']['reset_value']}"])) {
                $undo = $_lang[$module]["{$_form['form_params']['reset_value']}"];
            }
            else {
                $undo = (isset($_lang['jrCore'][9])) ? $_lang['jrCore'][9] : 'reset';
            }
        }
        $cancel_text = false;
        $cancel_url  = false;
        if (!empty($_form['form_params']['cancel'])) {

            // Cancel text
            if (isset($_form['form_params']['cancel_value']) && isset($_lang[$module]["{$_form['form_params']['cancel_value']}"])) {
                $cancel_text = $_lang[$module]["{$_form['form_params']['cancel_value']}"];
            }
            elseif (!empty($_form['form_params']['cancel_value'])) {
                $cancel_text = $_form['form_params']['cancel_value'];
            }
            else {
                $cancel_text = (isset($_lang['jrCore'][2])) ? $_lang['jrCore'][2] : 'cancel';
            }

            // Cancel Url
            if ($_form['form_params']['cancel'] == 'referrer') {
                $cancel_url = jrCore_get_local_referrer();
            }
            elseif ($_form['form_params']['cancel'] == 'modal_close') {
                $cancel_url = '$.modal.close();';
            }
            else {
                $cancel_url = $_form['form_params']['cancel'];
            }
        }
        // get lang replacements in place
        if (isset($_form['form_params']['submit_value'])) {
            if (isset($_lang[$module]["{$_form['form_params']['submit_value']}"])) {
                $_form['form_params']['submit_value'] = $_lang[$module]["{$_form['form_params']['submit_value']}"];
            }
        }
        else {
            $_form['form_params']['submit_value'] = '';
        }

        if (jrCore_get_user_session_key('quota_max_items_reached')) {
            jrCore_page_cancel_button($cancel_url, $cancel_text);
        }
        else {
            $form_validate = true;
            if (isset($_form['form_params']['form_validate']) && jrCore_checktype($_form['form_params']['form_validate'], 'is_false')) {
                $form_validate = false;
            }
            jrCore_form_submit($_form['form_params']['submit_value'], $undo, $cancel_text, $cancel_url, $form_validate);
        }
        jrCore_form_end();

        // Lastly - save all fields that rolled out on this form to the form session
        if (isset($_form['form_fields']) && is_array($_form['form_fields'])) {
            jrCore_form_update_session_fields($_form['form_token'], $_form['form_fields']);
        }

        // We only ever show a form once per page display
        jrCore_set_flag("jrcore_page_display_form_{$_form['form_token']}", 1);
    }

    $_tmp = jrCore_get_flag('jrcore_page_elements');

    // $_tmp['page'] contains all the page elements we are going to be showing on
    // this view - if we are a designer form, we need to adjust our field order here
    if ($design && isset($_tmp['page']) && is_array($_tmp['page'])) {

        $_or = array();
        $_nw = array();
        $num = 1;
        $elm = 0;
        $tab = -100;

        foreach ($_tmp['page'] as $k => $_field) {

            // Make sure field is active
            if (isset($_field['active']) && $_field['active'] == '0') {
                unset($_tmp['page'][$k]);
                continue;
            }

            // Make sure the user has permissions to view this field
            if (isset($_field['group'])) {
                if (!jrCore_user_is_part_of_group($_field['group'])) {
                    unset($_tmp['page'][$k]);
                    continue;
                }
            }

            // We need to check here for form fields.  ALL form fields must
            // come after the opening form element - so we first must scan for our
            // opening form element and make sure it comes before the form fields.
            if (isset($_field['name'])) {
                // If this is a "no form designer field" we come last
                if (isset($_field['form_designer']) && $_field['form_designer'] === false) {
                    if (!isset($_field['order'])) {
                        $val = (8000 + $elm);
                    }
                    else {
                        $val = (intval($_field['order']) * 100);
                    }
                }
                else {
                    if (strpos(' ' . $_field['name'], 'user_is_human') && !empty($_sfd['user_is_human']['order'])) {
                        // Form Designer value (in $_sfd)
                        $val = (int) $_sfd['user_is_human']['order'];
                    }
                    elseif (isset($_sfd["{$_field['name']}"]['order'])) {
                        // Form Designer value (in $_sfd) is FIRST
                        $val = (int) $_sfd["{$_field['name']}"]['order'];
                    }
                    elseif (isset($_field['order'])) {
                        // Defined in controller is SECOND
                        $val = (float) $_field['order'];
                    }
                    else {
                        $val = $k;
                    }
                    $val = ($val * 100);
                }
                if (in_array($val, $_or)) {
                    $val += 1;
                }
                $_or[$num] = $val;
                $elm       += 100;
            }
            else {
                switch ($_field['type']) {
                    case 'page_tab_bar':
                        // Tabs always appear at the top
                        $_or[$num] = $tab++;
                        break;
                    case 'page_banner':
                        $_or[$num] = -1;
                        break;
                    case 'form_submit':
                        $_or[$num] = 100000;
                        break;
                    case 'form_begin':
                    case 'page_notice':
                        $_or[$num] = 0;
                        break;
                    default:
                        $elm += 100;
                        if (in_array($elm, $_or)) {
                            $elm += 1;
                        }
                        $_or[$num] = $elm;
                        break;
                }
            }
            $_nw[$num] = $_field;
            $num++;
        }

        $_fn = array();
        if (count($_nw) > 0) {
            asort($_or, SORT_NUMERIC);
            $ti = 1;
            foreach ($_or as $k => $num) {
                // Fix tab index order
                if (isset($_nw[$k]['name']) && isset($_nw[$k]['html']) && strpos($_nw[$k]['html'], 'tabindex')) {
                    $_nw[$k]['html'] = preg_replace('/tabindex="[0-9]*"/', 'tabindex="' . $ti . '"', $_nw[$k]['html']);
                    $ti++;
                }
                $_fn[] = $_nw[$k];
                unset($_nw[$k]);

            }
            $_tmp['page'] = $_fn;
        }
    }

    //--------------------------------------
    // PROCESS VIEW
    //--------------------------------------

    $page = '';
    // STEP 1) Process all elements that will be on the page
    // We have to check for our form begin/end - they need to sit outside
    // of any tables or page elements.
    if (!empty($_tmp['form_begin'])) {
        $page .= $_tmp['form_begin'];
    }

    // Any hidden form elements need to follow the form_begin
    if (isset($_tmp['form_hidden']) && is_array($_tmp['form_hidden'])) {
        $page .= implode('', $_tmp['form_hidden']);
    }

    // Bring in page begin
    if (!$pcls = jrCore_get_flag('jrcore_page_class')) {
        $pcls = '';
    }
    else {
        $pcls = " {$pcls}";
    }
    $_rep = array('page_class' => $pcls);
    $page .= jrCore_parse_template('page_begin.tpl', $_rep, 'jrCore');

    // If we are a master admin user viewing Global or Quota config, we need to
    // set a flag so the "updated" time and default button show in help
    $show_update = '0';
    if (jrUser_is_master() && $_post['option'] == 'admin' && isset($_post['_1'])) {
        switch ($_post['_1']) {
            case 'global':
            case 'quota':
                $show_update = '1';
                break;
        }
    }

    // Parse form/page elements
    $_seen = array();
    $_sec  = array();
    if (isset($_tmp['page']) && is_array($_tmp['page'])) {
        foreach ($_tmp['page'] as $k => $_element) {

            // Make sure we only ever show each "name" once
            // this should never happen, but we've seen modules that
            // are not handling custom form insertions properly where
            // this could be the case
            if (isset($_element['name'])) {
                if (jrCore_get_user_session_key('quota_max_items_reached')) {
                    continue;
                }
                if (isset($_seen["{$_element['name']}"])) {
                    unset($_tmp['page'][$k]);
                    continue;
                }
                $_seen["{$_element['name']}"] = 1;
            }
            $_element['show_update_in_help'] = $show_update;
            if (!isset($_element['module'])) {
                jrCore_logger('CRI', "core: page element added without module set", $_element);
            }

            // For some element types we need to set the "default_label"
            switch ($_element['type']) {
                case 'select':
                    $_element['default_label'] = (isset($_element['default']) && isset($_element['options']["{$_element['default']}"])) ? $_element['options']["{$_element['default']}"] : false;
                    break;
                case 'editor':
                    // If this is a mobile device, and we are asking for an editor, we use a textarea instead
                    $_element['type'] = 'textarea';
                    break;
                case 'page_section_header':
                    // We only show section headers 1 time
                    if (isset($_element['section'])) {
                        $_sec["{$_element['section']}"] = 1;
                    }
                    break;
            }

            // Setup our default value properly for display
            $_element['default_value'] = '';
            if (isset($_element['default']) && is_string($_element['default'])) {
                $_element['default_value'] = str_replace(array("\r\n", "\r", "\n"), '\n', addslashes($_element['default']));
            }

            // Check for section
            if (isset($_element['section']) && strlen($_element['section']) > 0 && !isset($_sec["{$_element['section']}"])) {
                $page                           .= jrCore_parse_template('page_section_header.tpl', array('title' => $_element['section']), 'jrCore');
                $_sec["{$_element['section']}"] = 1;
            }

            // Check for template - if we have a template, render it - else use HTML that comes from function
            if (!empty($_element['template'])) {
                $page .= jrCore_parse_template($_element['template'], $_element, 'jrCore');
            }
            elseif (!empty($_element['html'])) {
                $page .= $_element['html'];
            }
        }
    }
    jrCore_delete_user_session_key('quota_max_items_reached');
    unset($_seen);

    // Bring in page end
    $page .= jrCore_parse_template('page_end.tpl', $_rep, 'jrCore');

    // We have to check for our form begin/end - they need to sit outside
    // of any tables or page elements.
    if (isset($_tmp['form_end'])) {
        $page .= $_tmp['form_end'];
    }

    // as well as modal window HTML
    if (isset($_tmp['form_modal'])) {
        $page .= jrCore_parse_template($_tmp['form_modal']['template'], array_merge($_rep, $_tmp['form_modal']), 'jrCore');
    }

    // STEP 2) Process HEADER
    // @note: The header is processed AFTER we process page elements, since the page elements themselves
    // can include new jrcore_page_elements that we need when processing the HEADER
    $html = '';
    $meta = false;
    $_tmp = jrCore_get_flag('jrcore_page_elements');
    $temp = jrCore_get_flag('jrcore_page_no_header_or_footer');
    if (!$temp) {
        // Check for META header only
        $meta = jrCore_get_flag('jrcore_page_meta_header_only');
        if ($meta) {
            $html .= jrCore_parse_template('meta.tpl', $_tmp) . "\n";
            if (!empty($body_id)) {
                $html .= "<body id=\"" . $body_id . '">';
            }
            else {
                $html .= '<body>';
            }
        }
        else {
            $html .= jrCore_parse_template('header.tpl', $_tmp);
        }
    }
    else {
        // With no header being shown, any form elements added in that
        // need JS/CSS added to the meta will not function properly.
        if (!jrCore_get_flag('jrcore_page_defer_js_and_css')) {
            if ($_mta = jrCore_get_flag('jrcore_page_elements')) {
                $test = serialize($_mta);
                if (strpos($test, 'javascript_') || strpos($test, 'css_')) {
                    $html .= jrCore_parse_template('meta_javascript_and_css.tpl', $_mta, 'jrCore');
                }
            }
        }
        else {
            // We are deferring CSS - this means that we are NOT going to add
            // JS or CSS into this view since our OUTPUT is going to be processed
            // and added to another page where the JS and CSS will be included
            // So we UNSET this flag here - our next call to jrCore_page_display
            // will no longer have this flag and the JS and CSS will be included
            jrCore_delete_flag('jrcore_page_defer_js_and_css');
        }
    }

    // See if we are doing an ADMIN MENU VIEW for this module/view
    $dash = jrCore_get_flag('jrcore_dashboard_active');
    if ($admn && isset($_adm)) {
        $_adm['admin_page_content'] = $page;
        $html                       .= jrCore_parse_template('admin.tpl', $_adm, 'jrCore');
    }
    elseif ($dash) {
        $_rep = array('dashboard_html' => $page);
        $html .= jrCore_parse_template('dashboard.tpl', $_rep, 'jrCore');
    }
    else {
        $html .= $page;
    }

    // STEP 3) Process FOOTER
    if (!$temp) {
        // Check for META header only
        if ($meta) {
            $html .= "\n</body>";
        }
        else {
            $html .= jrCore_parse_template('footer.tpl', $_tmp);
        }
    }
    else {
        // Reset for next show
        jrCore_delete_flag('jrcore_page_no_header_or_footer');
    }

    // Delete page and form elements
    $_tmp = jrCore_get_flag('jrcore_page_elements');
    if ($_tmp) {
        foreach ($_tmp as $k => $v) {
            if (strpos($k, 'javascript') !== 0 && strpos($k, 'css') !== 0) {
                unset($_tmp[$k]);
            }
        }
        jrCore_set_flag('jrcore_page_elements', $_tmp);
    }

    if ($return_html) {
        return $html;
    }
    echo $html;
    return true;
}

/**
 * The jrCore_page_button function is used for generating the necessary "button"
 * HTML code in the Jamroom Control Panel.  This ensures the Control Panel buttons
 * can be styled via the form.tpl file.
 * form.tpl element name: form_button
 * @param string $id DOM ID for button
 * @param string $value Text value for button
 * @param string $onclick JS Onclick event function
 * @param array $_att Additional HTML attributes
 * @return string Returns HTML of button code
 */
function jrCore_page_button($id, $value, $onclick, $_att = null)
{
    // Check for provided class...
    $cls = 'form_button';
    if (isset($_att['class'])) {
        $cls = $_att['class'];
        unset($_att['class']);
    }
    $value = jrCore_entity_string(html_entity_decode($value, ENT_QUOTES));
    if ($onclick == 'disabled') {
        $html = '<input type="button" id="' . $id . '" class="' . $cls . ' form_button_disabled" name="' . $id . '" value="' . $value . '" disabled="disabled"';
    }
    else {
        $html = '<input type="button" id="' . $id . '" class="' . $cls . '" name="' . $id . '" value="' . $value . '" onclick="' . $onclick . '"';
    }
    if (is_array($_att)) {
        foreach ($_att as $key => $attr) {
            $html .= ' ' . $key . '="' . $attr . '"';
        }
    }
    $html .= '>';
    return $html;
}

/**
 * Include the CSS for the Material Design Icons
 * @return bool
 */
function jrCore_page_include_icon_button_css()
{
    if (!jrCore_get_flag('material_design_css_included')) {
        // What style are we doing? "outlined" is default
        $class = 'material-icons-outlined';
        $style = '+Outlined';
        if ($_tmp = jrCore_get_registered_module_features('jrCore', 'icon_style')) {
            $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
            if (isset($_tmp[$skin])) {
                $value = array_keys($_tmp[$skin]);
                $value = reset($value);
                switch ($value) {
                    case 'two-tone':
                        $class = 'material-icons-two-tone';
                        $style = '+Two+Tone';
                        break;
                    case 'round':
                        $class = 'material-icons-round';
                        $style = '+Round';
                        break;
                    case 'sharp':
                        $class = 'material-icons-sharp';
                        $style = '+Sharp';
                        break;
                    case 'filled':
                        $class = 'material-icons';
                        $style = '';
                        break;
                }
            }
        }
        $_css = array('source' => "https://fonts.googleapis.com/icon?family=Material+Icons{$style}");
        jrCore_create_page_element('css_href', $_css);
        jrCore_set_flag('material_design_css_included', $class);
    }
    return true;
}

/**
 * Create a page button using a Material Design icon
 * @link https://material.io/resources/icons
 * @param string $id DOM ID for button
 * @param string $icon Material Design Icon
 * @param string $onclick JS Onclick event function
 * @param array $_att Additional HTML attributes
 * @return string Returns HTML of button code
 */
function jrCore_page_icon_button($id, $icon, $onclick, $_att = null)
{
    $cls = 'form_icon_button';
    if (!empty($_att['class'])) {
        $cls = $_att['class'];
        unset($_att['class']);
    }
    jrCore_page_include_icon_button_css();
    $class = jrCore_get_flag('material_design_css_included');
    $icon  = "<i class=\"{$class} form_button_icon form_button_icon_{$icon}\">{$icon}</i>";
    if ($onclick == 'disabled') {
        $cls .= " form_button_disabled";
    }
    $html = '<div id="' . $id . '" class="' . $cls . '"';
    if ($onclick != 'disabled') {
        $html .= ' onclick="' . $onclick . '"';
    }
    if (is_array($_att)) {
        foreach ($_att as $key => $attr) {
            $html .= ' ' . $key . '="' . $attr . '"';
        }
    }
    $html .= '>' . $icon . '</div>';
    return $html;
}

/**
 * Creates a "button menu" that when clicked drops down a selection menu
 * @param string $name Value for Button
 * @param array $_buttons links that will appear in the button
 * @param int $size
 * @return string Returns HTML of button code
 */
function jrCore_page_button_menu($name, $_buttons, $size = null)
{
    if (!is_array($_buttons)) {
        return false;
    }
    if (is_null($size)) {
        $size = jrCore_get_active_skin_icon_size();
    }
    if (empty($_buttons['icon']) || $_buttons['icon'] == 'dot-column') {
        $icon = jrCore_get_cached_svg('jrCore', 'menu.svg', $size);
    }
    elseif ($_buttons['icon'] == 'dot-column-disabled') {
        $icon = jrCore_get_cached_svg('jrCore', 'menu.svg', $size, 'svg_icon svg_icon_disabled');
        return '<div class="form_button_menu" id="' . $name . '"><a class="form_button_menu_button">' . $icon . '</a></div>';
    }
    else {
        $icon = jrCore_get_icon_html($_buttons['icon'], $size);
    }
    $label = '';
    if (isset($_buttons['label'])) {
        $label = jrCore_entity_string(html_entity_decode($_buttons['label'], ENT_QUOTES));
    }
    // Check for provided class...
    $html = '<div class="form_button_menu" id="' . $name . '" onclick="jrCore_toggle_button_menu(\'' . $name . '\');event.stopPropagation();"><a class="form_button_menu_button">' . $icon . ' ' . $label . '</a>
    <ul class="form_button_menu_ul">';
    foreach ($_buttons['_link'] as $_lnk) {
        $id    = (isset($_lnk['id'])) ? "id=\"{$_lnk['id']}\"" : '';
        $class = (isset($_lnk['class'])) ? $_lnk['class'] : '';
        $html  .= ' <li ' . $id . ' class="form_button_submenu ' . $class . '" onclick="' . $_lnk['onclick'] . '"> ' . $_lnk['value'] . ' </li> ';
    }
    $html .= ' </ul></div>';
    if (!jrCore_get_flag('page_menu_button_close')) {
        $_js = array("$(document).on('click',function(){ $('.form_button_menu_ul').hide() });");
        jrCore_create_page_element('javascript_ready_function', $_js);
        jrCore_set_flag('page_menu_button_close', 1);
    }
    return $html;
}

/**
 * Show a pending notice in a form
 * @param string $module Module
 * @param array $_item Item info
 * @param bool $return_output set to TRUE to return button/message output
 * @return bool
 */
function jrCore_show_pending_notice($module, $_item, $return_output = false)
{
    global $_conf;
    $prefix = jrCore_db_get_prefix($module);
    if ($module != 'jrUser' && (!isset($_item["{$prefix}_pending"]) || intval($_item["{$prefix}_pending"]) === 0)) {
        // This item is NOT pending
        return true;
    }
    if ($module == 'jrUser') {
        if (isset($_item['user_validated']) && $_item['user_validated'] == '0') {
            $_item["{$prefix}_pending"] = 1;
        }
        else {
            // User is already validated...
            return true;
        }
        $tag = 'user';
    }
    else {
        $tag = 'item';
    }

    // Are we coming from the dashboard?
    $ref = jrCore_get_local_referrer();
    if (strpos($ref, '/dashboard/')) {
        $_SESSION['user_dashboard_referral_url'] = $ref;
    }

    $pnd = (int) $_item["{$prefix}_pending"];
    // We are pending - show notice to normal users, approval options to admin users
    if (jrUser_is_admin()) {

        // Pending Status
        // 1 = Pending Approval
        // 2 = Pending Approval - Rejected
        switch ($pnd) {
            case 2:
                $out = "This {$tag} was <b>rejected</b> and is not visible to users until approved.";
                // Do we have out reason(s) saved on this one?
                if (isset($_item["{$prefix}_pending_reason"]) && strlen($_item["{$prefix}_pending_reason"]) > 0) {
                    $out .= "<br>" . nl2br($_item["{$prefix}_pending_reason"]);
                }
                $out .= '<br><br>';
                $shw = false;
                break;
            default:
                $out = "This {$tag} is currently <b>pending approval</b> and will be visible to other users once approved.<br><br>";
                $shw = true;
                break;
        }

        $url = jrCore_get_module_url('jrCore');
        if ($module == 'jrUser') {
            $srl = jrCore_get_module_url('jrUser');
            $out .= jrCore_page_button("approve", 'approve', "jrCore_confirm('Activate User Account?', 'This will activate the User Account and send them an email', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$srl}/user_activate/user_id={$_item['_user_id']}') })") . '&nbsp;';
            $out .= jrCore_page_button("delete", 'delete', "jrCore_confirm('Delete User Account?', 'NOTE - This will NOT delete the User Profile associated with the account', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$srl}/delete_save/id={$_item['_user_id']}') })");
        }
        else {
            $out .= jrCore_page_button('approve', 'approve', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_approve/{$module}/id={$_item['_item_id']}')") . '&nbsp';
            if ($shw) {
                $out .= jrCore_page_button('reject', 'reject', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_reject/{$module}/id={$_item['_item_id']}')") . '&nbsp';
            }
            $out .= jrCore_page_button('delete', 'delete', "jrCore_confirm('Delete this {$tag}?', 'No notification will be sent', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_delete/{$module}/id={$_item['_item_id']}')})");
        }
        if ($return_output) {
            return $out;
        }
        jrCore_page_notice('notice', $out, false);
    }
    else {

        $_lang = jrUser_load_lang_strings();
        if ($module == 'jrUser' && !jrUser_is_admin()) {
            // We already tell them if their account is pending
            return true;
        }
        // Pending Status
        // 1 = Pending Approval
        // 2 = Pending Approval - Rejected
        switch ($pnd) {
            case 2:
                if ($return_output) {
                    return $_lang['jrCore'][99];
                }
                $not = $_lang['jrCore'][99];
                // Do we have out reason(s) saved on this one?
                if (isset($_item["{$prefix}_pending_reason"]) && strlen($_item["{$prefix}_pending_reason"]) > 0) {
                    $not .= "<br>" . nl2br($_item["{$prefix}_pending_reason"]);
                }
                break;

            default:
                if ($return_output) {
                    return $_lang['jrCore'][71];
                }
                $not = $_lang['jrCore'][71];
                break;
        }
        jrCore_page_notice('error', $not, false);
    }
    return true;
}

/**
 * @param $inline bool set to TRUE to return inline CSS / JS
 * @return bool
 * @deprecated
 * Add CodeMirror syntax highlighting Javascript src URLs to a page
 */
function jrCore_add_code_mirror_js($inline = false)
{
    if ($inline) {
        if (!jrCore_get_flag('jrcore_code_mirror_inline_added')) {
            $out = "<style>\n";
            $out .= jrCore_file_get_contents(APP_DIR . '/modules/jrCore/contrib/codemirror/lib/codemirror.css');
            $out .= "</style>\n<script>\n";
            $_sr = array('lib/codemirror.js', 'mode/htmlmixed/htmlmixed.js', 'mode/xml/xml.js', 'mode/javascript/javascript.js', 'mode/css/css.js', 'mode/clike/clike.js', 'mode/php/php.js', 'mode/smarty/smarty.js');
            foreach ($_sr as $src) {
                $out .= jrCore_file_get_contents(APP_DIR . "/modules/jrCore/contrib/codemirror/{$src}");
            }
            $out .= '</script>';
            jrCore_set_flag('jrcore_code_mirror_inline_added', 1);
            return $out;
        }
    }
    else {
        if (!jrCore_get_flag('jrcore_code_mirror_js_added')) {
            $url = jrCore_get_base_url();
            jrCore_create_page_element('css_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/lib/codemirror.css"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/lib/codemirror.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/htmlmixed/htmlmixed.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/xml/xml.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/javascript/javascript.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/css/css.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/clike/clike.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/php/php.js"));
            jrCore_create_page_element('javascript_footer_href', array('source' => "{$url}/modules/jrCore/contrib/codemirror/mode/smarty/smarty.js"));
            jrCore_set_flag('jrcore_code_mirror_js_added', 1);
        }
    }
    return true;
}
