<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Temp and Cache
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

//-------------------------------------------
// Local Cache
//-------------------------------------------

/**
 * Check if local caching is enabled
 * @return bool
 */
function jrCore_local_cache_is_enabled()
{
    return function_exists('apcu_add');
}

/**
 * Get local cache key prefix
 * @param bool $is_cache set to FALSE if the key is not a cache entry
 * @return string
 */
function jrCore_local_cache_get_key_prefix($is_cache)
{
    // @note: Do not use $_conf in this function - not always set
    // @note: the 'jrcore_config_and_modules_key' flag will be UNIQUE based on site config
    $pfx = ($is_cache) ? '' : '_';
    return $pfx . substr(jrCore_get_flag('jrcore_config_and_modules_key'), 0, 5);
}

/**
 * Add a key and value to the local cache
 * @param string $key Key Name
 * @param mixed $value Key Value
 * @param int $ttl Number of seconds to cache key for
 * @param bool $is_cache set to FALSE if the key is not a cache entry
 * @return array|bool
 */
function jrCore_set_local_cache_key($key, $value, $ttl = 0, $is_cache = true)
{
    if (jrCore_local_cache_is_enabled()) {
        if ($is_cache && jrCore_is_developer_mode()) {
            return true;
        }
        $pfx = jrCore_local_cache_get_key_prefix($is_cache);
        return apcu_store("{$pfx}:{$key}", $value, $ttl);
    }
    return false;
}

/**
 * Get a locally cached key value
 * @param string $key
 * @param bool $is_cache set to FALSE if the key is not a cache entry
 * @return mixed
 */
function jrCore_get_local_cache_key($key, $is_cache = true)
{
    if (jrCore_local_cache_is_enabled()) {
        if ($is_cache && jrCore_is_developer_mode()) {
            return false;
        }
        $pfx = jrCore_local_cache_get_key_prefix($is_cache);
        return apcu_fetch("{$pfx}:{$key}");
    }
    return false;
}

/**
 * Delete a locally cached key value
 * @param string $key
 * @param bool $is_cache set to FALSE if the key is not a cache entry
 * @return bool
 */
function jrCore_delete_local_cache_key($key, $is_cache = true)
{
    if (jrCore_local_cache_is_enabled()) {
        $pfx = jrCore_local_cache_get_key_prefix($is_cache);
        return apcu_delete("{$pfx}:{$key}");
    }
    return true;
}

/**
 * Delete local APCu cache entries
 * @param bool $full_reset Set to TRUE to fully reset the entire APCu cache (instead of just items marked as CACHE)
 * @note: only deletes entries that have been marked as CACHE ($is_cache = TRUE)
 * @return bool
 */
function jrCore_reset_local_cache($full_reset = false)
{
    if (jrCore_local_cache_is_enabled()) {
        if ($full_reset) {
            // Reset entire APCu cache - this prevents fragmentation
            jrCore_start_timer('apcu_maintenance');
            apcu_clear_cache();
            jrCore_stop_timer('apcu_maintenance');
        }
        else {
            jrCore_start_timer('apcu_maintenance');
            if ($_tmp = apcu_cache_info()) {
                if (isset($_tmp['cache_list']) && is_array($_tmp['cache_list'])) {
                    $old = substr(jrCore_get_config_value('jrCore', 'unique_string', '0'), 0, 5);  // Pre Core 6.5.0 prefix
                    $pfx = jrCore_local_cache_get_key_prefix(true);
                    foreach ($_tmp['cache_list'] as $c) {
                        if (strpos($c['info'], "{$pfx}:") === 0 || strpos($c['info'], "{$old}:") === 0) {
                            apcu_delete($c['info']);
                        }
                    }
                }
            }
            jrCore_stop_timer('apcu_maintenance');
        }
    }
    return true;
}

/**
 * Increment a local cache key
 * @param string $key
 * @param int $count amount to increment
 * @return bool
 */
function jrCore_increment_local_cache_key($key, $count = 1)
{
    if (jrCore_local_cache_is_enabled()) {
        $pfx = jrCore_local_cache_get_key_prefix(false);
        $key = "{$pfx}:{$key}";
        return apcu_inc($key, intval($count));
    }
    return false;
}

/**
 * Decrement a local cache key
 * @param string $key
 * @param int $count amount to increment
 * @return bool
 */
function jrCore_decrement_local_cache_key($key, $count = 1)
{
    if (jrCore_local_cache_is_enabled()) {
        $pfx = jrCore_local_cache_get_key_prefix(false);
        $key = "{$pfx}:{$key}";
        if (apcu_exists($key)) {
            return apcu_dec($key, intval($count));
        }
    }
    return false;
}

/**
 * Get APC memory fragmentation percent
 * @return bool|float|int
 */
function jrCore_get_local_cache_fragmentation_percent()
{
    if (jrCore_local_cache_is_enabled()) {
        jrCore_start_timer('apcu_maintenance');
        if ($_apc = apcu_sma_info()) {
            jrCore_stop_timer('apcu_maintenance');
            $num_seg    = 0;
            $free_seg   = 0;
            $frag_size  = 0;
            $free_total = 0;
            for ($i = 0; $i < $_apc['num_seg']; $i++) {
                $ptr = 0;
                foreach ($_apc['block_lists'][$i] as $block) {
                    if ($block['offset'] != $ptr) {
                        ++$num_seg;
                    }
                    $ptr = $block['offset'] + $block['size'];
                    // Only consider blocks < 5M for the fragmentation %
                    if ($block['size'] < (5 * 1024 * 1024)) {
                        $frag_size += $block['size'];
                    }
                    $free_total += $block['size'];
                }
                $free_seg += count($_apc['block_lists'][$i]);
            }
            if ($free_seg > 1) {
                return round(($frag_size / $free_total) * 100, 1);
            }
            return 0;
        }
        jrCore_stop_timer('apcu_maintenance');
    }
    return false;
}

//-------------------------------------------
// Temp Value
//-------------------------------------------

/**
 * Temp: Save a Temp Value to the Temp DB
 * The jrCore_set_temp_value function will store a "value" for a "key"
 * for a given module.  It is guaranteed to NOT conflict with any other
 * module on the system (including the core).  This value can be retrieved
 * at a later point using jrCore_get_temp_value.
 * @param string $module Module to store temp value for
 * @param string $key Key unique key for temp value (max length = 128)
 * @param mixed $value Value to store - can be string or array
 * @return bool
 */
function jrCore_set_temp_value($module, $key, $value)
{
    if (is_string($key)) {
        $key = substr($key, 0, 128);
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "INSERT INTO {$tbl} (temp_module,temp_key,temp_updated,temp_value)
                VALUES ('" . jrCore_db_escape($module) . "','" . jrCore_db_escape($key) . "',UNIX_TIMESTAMP(),'" . jrCore_db_escape(json_encode($value)) . "')
                ON DUPLICATE KEY UPDATE temp_updated = UNIX_TIMESTAMP(),temp_value = VALUES(temp_value)";
        $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
        // 0 = no update
        // 1 = new row inserted
        // 2 = row updated
        if ($cnt && is_numeric($cnt)) {
            return true;
        }
    }
    return false;
}

/**
 * Temp: Update a Temp Value in the Temp DB
 * The jrCore_update_temp_value function will return a "value" that has been
 * stored in the Temp Table, when given the KEY.  This is guaranteed to
 * be unique to the calling module.
 * @param string $module Module that saved the temp value
 * @param string $key Key existing key to update
 * @param mixed $value New Value to store - can be string or array
 * @return bool
 */
function jrCore_update_temp_value($module, $key, $value)
{
    if (is_string($key)) {
        $mod = jrCore_db_escape($module);
        $key = jrCore_db_escape($key);
        $val = jrCore_db_escape(json_encode($value));
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "UPDATE {$tbl} SET temp_updated = UNIX_TIMESTAMP(),temp_value = '{$val}' WHERE temp_module = '{$mod}' AND temp_key = '{$key}' LIMIT 1";
        $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
        if ($cnt === 1) {
            return true;
        }
    }
    return false;
}

/**
 * Temp: Get a Temp Value from the Temp DB
 * The jrCore_get_temp_value function will return a "value" that has been
 * stored in the Temp Table, when given the KEY.  This is guaranteed to
 * be unique to the calling module.
 * @param string $module Module that saved the temp value
 * @param mixed $key Key to retrieve
 * @return mixed Returns value (string or array) on success, bool false on key does not exist
 */
function jrCore_get_temp_value($module, $key)
{
    if (is_string($key)) {
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "SELECT temp_value FROM {$tbl} WHERE temp_module = '" . jrCore_db_escape($module) . "' AND temp_key = '" . jrCore_db_escape($key) . "' LIMIT 1";
        $_rt = jrCore_db_query($req, 'SINGLE', false, null, false);
        if ($_rt && is_array($_rt) && !empty($_rt['temp_value'])) {
            return json_decode($_rt['temp_value'], true);
        }
    }
    return false;
}

/**
 * Temp: Delete an Existing Temp Value in the Temp DB
 * The jrCore_delete_temp_value function will delete a temp value that was
 * previously set by the jrCore_set_temp_value function.
 * @param string $module Module that saved the temp value
 * @param string $key Key to delete
 * @return bool returns true if key is deleted, false if key is not found
 */
function jrCore_delete_temp_value($module, $key)
{
    if (is_string($key)) {
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "DELETE FROM {$tbl} WHERE temp_module = '" . jrCore_db_escape($module) . "' AND temp_key = '" . jrCore_db_escape($key) . "' LIMIT 1";
        $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
        if ($cnt === 1) {
            return true;
        }
    }
    return false;
}

/**
 * Temp: Cleanup old Temp Values in the Temp DB
 * @param string $module Module that saved the temp value
 * @param int $safe_seconds number of seconds where of an entry is GREATER than it will be removed
 * @return int Returns number of Temp entries deleted
 */
function jrCore_clean_temp($module, $safe_seconds)
{
    if (jrCore_checktype($safe_seconds, 'number_nn')) {
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "DELETE FROM {$tbl} WHERE temp_module = '" . jrCore_db_escape($module) . "' AND temp_updated <= (UNIX_TIMESTAMP() - {$safe_seconds})";
        return jrCore_db_query($req, 'COUNT', false, null, false);
    }
    return false;
}

/**
 * Temp: Get all values for a module
 * @param string $module Module that saved the temp values
 * @return array|false
 * @since 6.5.0b9
 */
function jrCore_get_all_temp_values($module)
{
    $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
    $req = "SELECT temp_key, temp_value FROM {$tbl} WHERE temp_module = '" . jrCore_db_escape($module) . "'";
    $_rt = jrCore_db_query($req, 'temp_key', false, 'temp_value');
    if ($_rt && is_array($_rt)) {
        $_tm = array();
        foreach ($_rt as $k => $v) {
            $_tm[$k] = json_decode($v, true);
        }
        return $_tm;
    }
    return false;
}

//-------------------------------------------
// Cache System
//-------------------------------------------

/**
 * Reset sprite caches
 * @return bool
 */
function jrCore_reset_sprite_cache()
{
    global $_conf;
    $dir = jrCore_get_module_cache_dir($_conf['jrCore_active_skin']);
    jrCore_start_timer('filesystem');
    $_fl = glob("{$dir}/*sprite*", GLOB_NOSORT);
    if ($_fl && is_array($_fl)) {
        foreach ($_fl as $file) {
            unlink($file);  // OK
        }
    }
    $dir = jrCore_get_media_directory(0, FORCE_LOCAL);
    $_fl = glob("{$dir}/*sprite*", GLOB_NOSORT);
    if ($_fl && is_array($_fl)) {
        foreach ($_fl as $file) {
            unlink($file);  // OK
        }
    }
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Reset cached template files
 */
function jrCore_reset_template_cache()
{
    // When resetting template caches we delete:
    // - ALL JS and CSS files
    // - ALL file and string templates
    jrCore_start_timer('filesystem');
    $_pat = array('tpl.php', 'string.php');
    $_tmp = array();
    foreach ($_pat as $pattern) {
        $_tmp = array_merge($_tmp, glob(APP_DIR . "/data/cache/*/*.{$pattern}", GLOB_NOSORT));
    }
    if (count($_tmp) > 0) {
        foreach ($_tmp as $file) {
            $file = realpath($file);
            if (strpos($file, APP_DIR . '/data/cache/') === 0) {
                unlink($file);  // OK
                opcache_invalidate($file, true);
            }
        }
    }
    jrCore_stop_timer('filesystem');

    // Reset APCu template cache as well
    // @note: jrCore_reset_local_cache() will reset the template
    // cache as well, but this is needed here in case the local
    // cache is NOT reset, but the template cache is
    if (jrCore_local_cache_is_enabled()) {
        if ($_tmp = apcu_cache_info()) {
            if (isset($_tmp['cache_list']) && is_array($_tmp['cache_list'])) {
                $old = substr(jrCore_get_config_value('jrCore', 'unique_string', '0'), 0, 5);
                $pfx = jrCore_local_cache_get_key_prefix(true);
                foreach ($_tmp['cache_list'] as $c) {
                    if (strpos($c['info'], "{$pfx}:t:") === 0 || strpos($c['info'], "{$old}:t:") === 0) {
                        apcu_delete($c['info']);
                    }
                }
            }
        }
    }
    return true;
}

/**
 * Reset list of compiled smarty templates
 * @param array $_files List of template files to reset
 * @return bool
 */
function jrCore_reset_template_cache_files($_files)
{
    // jrCore^activity_indicator.tpl.php
    // jrCore^form_field_elements.tpl.php
    jrCore_start_timer('filesystem');
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if ($_tpl = glob(APP_DIR . "/data/cache/{$skin}/*.php")) {
        foreach ($_tpl as $file) {
            foreach ($_files as $cache_file) {
                if (strpos($file, "{$cache_file}^") && strpos($file, APP_DIR . '/data/cache/') === 0) {
                    unlink($file);  // OK
                    opcache_invalidate($file, true);
                }
            }
        }
    }
    jrCore_stop_timer('filesystem');

    // Reset APCu template cache as well
    // @note: jrCore_reset_local_cache() will reset the template
    // cache as well, but this is needed here in case the local
    // cache is NOT reset, but the template cache is
    if (jrCore_local_cache_is_enabled()) {
        foreach ($_files as $cache_file) {
            list($mod, $tpl) = explode('_', $cache_file, 2);
            $tpl = str_replace('_tpl', '.tpl', $tpl);
            jrCore_reset_apc_template_file($tpl, $mod);
        }
    }
    return true;
}

/**
 * Check if caching is enabled
 * @return bool
 */
function jrCore_caching_is_enabled()
{
    // Check to see if we are enabled
    if (jrCore_get_config_value('jrCore', 'default_cache_seconds', 300) === 0) {
        return false;
    }
    // Check for developer mode
    if (jrCore_is_developer_mode()) {
        return false;
    }
    $_data = jrCore_trigger_event('jrCore', 'caching_is_enabled', array('enabled' => true));
    if (is_array($_data) && isset($_data['enabled']) && is_bool($_data['enabled'])) {
        return $_data['enabled'];
    }
    return true;
}

/**
 * Get configured cache plugins
 * @return array
 */
function jrCore_get_cache_system_plugins()
{
    return jrCore_get_system_plugins('cache');
}

/**
 * Returns the active caching sub system
 * @return string
 */
function jrCore_get_active_cache_system()
{
    return jrCore_get_config_value('jrCore', 'active_cache_system', 'jrCore_mysql');
}

/**
 * Deletes the core Module and Config cache
 * @param bool $force set to TRUE to force immediate config reset and reload - otherwise happens at end of current process
 * @return bool
 */
function jrCore_delete_config_cache($force = false)
{
    // @note: Core config and settings is always stored in the database
    $key = jrCore_get_flag('jrcore_config_and_modules_key');
    jrCore_trigger_event('jrCore', 'delete_config_cache', array('jrcore_config_and_modules_key' => $key, 'force' => $force));
    _jrCore_mysql_delete_cache('jrCore', $key);
    return jrCore_delete_local_cache_key('config_cache');
}

/**
 * Delete all cache entries
 * @param string $module Optionally delete all cache entries for a specific module
 * @param int $user_id Optionally delete all cache entries for specific User ID
 * @return bool
 */
function jrCore_delete_all_cache_entries($module = null, $user_id = null)
{
    // Settings are ALWAYS cached - regardless of cache setting
    $temp = jrCore_get_active_cache_system();
    if (is_null($module) && is_null($user_id)) {
        jrCore_delete_config_cache();
    }
    $func = "_{$temp}_delete_all_cache_entries";
    if (function_exists($func)) {
        // @note: This is OK since this is JUST this process
        jrCore_delete_all_cache_flags();
        return $func($module, $user_id);
    }
    jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
    return false;
}

/**
 * Delete all cache entries for a specific profile
 * @param int $profile_id Optionally delete all cache entries for specific User ID
 * @param string $module Optionally delete all cache entries for a specific module
 * @return bool
 */
function jrCore_delete_profile_cache_entries($profile_id = null, $module = null)
{
    global $_mods;
    if (!jrCore_caching_is_enabled()) {
        return true;
    }
    $pky = 'jrcore_process_exit_delete_profile_cache';
    if (!$_tm = jrCore_get_flag($pky)) {
        $_tm = array();
    }
    $pid = (int) $profile_id;
    if (!isset($_tm[$pid])) {
        $_tm[$pid] = (is_null($module) || !isset($_mods[$module])) ? 0 : $module;
    }
    elseif ($_tm[$pid] != 0 && is_null($module)) {
        $_tm[$pid] = 0;
    }
    jrCore_set_flag($pky, $_tm);
    return true;
}

/**
 * Delete profile cache on exit
 * @return bool
 */
function jrCore_process_exit_delete_profile_cache()
{
    if (!jrCore_caching_is_enabled()) {
        return true;
    }
    $key = 'jrcore_process_exit_delete_profile_cache';
    if ($_tm = jrCore_get_flag($key)) {
        $temp = jrCore_get_active_cache_system();
        $func = "_{$temp}_process_exit_delete_profile_cache";
        if (function_exists($func)) {
            if ($func($_tm)) {
                jrCore_delete_flag($key);
                return true;
            }
            return false;
        }
        jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
        return false;
    }
    return true;
}

/**
 * Cache: Delete cache for a given key
 * @param string $module Module to save cache for
 * @param string $key Key to delete cache for
 * @param bool $add_user DEPRECATED By default each cache entry is for a specific User ID - set to false to override
 * @param bool $add_skin DEPRECATED Add active skin name to unique cache key
 * @return bool
 */
function jrCore_delete_cache($module, $key, $add_user = true, $add_skin = true)
{
    return jrCore_delete_multiple_cache_entries(array(array($module, $key)));
}

/**
 * Cache: Delete multiple cache entries
 * @param $_items array consisting of 'module' (0) and 'key' (1)
 * @return bool
 */
function jrCore_delete_multiple_cache_entries($_items)
{
    if (!jrCore_caching_is_enabled()) {
        return true;
    }
    if (!is_array($_items) || count($_items) === 0) {
        // Nothing to delete
        return false;
    }
    $uid = 0;
    $add = false;
    if (jrUser_is_logged_in()) {
        $uid = (int) jrCore_get_user_session_key('_user_id');
        $add = true;
    }
    $temp = jrCore_get_active_cache_system();
    $func = "_{$temp}_delete_multiple_cache_entries";
    if (function_exists($func)) {
        $_keys = array();

        // 0 = module, 1 = key
        foreach ($_items as $i) {

            // We always reset the associated key for logged out users
            $key         = jrCore_get_cache_key($i[0], $i[1], 0, true);
            $_keys[$key] = array($i[0], $key);
            jrCore_delete_cache_flag($key);

            $key         = jrCore_get_cache_key($i[0], $i[1], 0, false);
            $_keys[$key] = array($i[0], $key);
            jrCore_delete_cache_flag($key);

            if ($add) {
                // This is a logged in user - reset for user
                $key         = jrCore_get_cache_key($i[0], $i[1], $uid, true);
                $_keys[$key] = array($i[0], $key);
                jrCore_delete_cache_flag($key);

                $key         = jrCore_get_cache_key($i[0], $i[1], $uid, false);
                $_keys[$key] = array($i[0], $key);
                jrCore_delete_cache_flag($key);
            }

        }
        return $func($_keys);
    }
    jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
    return false;
}

/**
 * Cache: Check if a given key is cached
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to override
 * @param bool $add_skin Add active skin name to unique cache key
 * @param bool $flag_cache set to FALSE to not store result with jrCore_get_cache_flag()
 * @param bool $add_user_info Set to FALSE to not add user_id, language and device type
 * @return mixed returns string on success, bool false on not cached
 */
function jrCore_is_cached($module, $key, $add_user = true, $add_skin = true, $flag_cache = true, $add_user_info = true)
{
    if (!jrCore_caching_is_enabled()) {
        return false;
    }
    $temp = jrCore_get_active_cache_system();
    $func = "_{$temp}_is_cached";
    if (function_exists($func)) {
        $uid = ($add_user && jrUser_is_logged_in()) ? intval(jrCore_get_user_session_key('_user_id')) : 0;
        // @note: skin name is always added in jrCore_get_cache_key()
        $key = jrCore_get_cache_key($module, $key, $uid, $add_user_info);
        if ($flag_cache) {
            if ($out = jrCore_get_cache_flag($key)) {
                if (is_array($out) && time() < $out[0]) {
                    // We're still good
                    return jrCore_replace_emoji($out[1]);
                }
                jrCore_delete_cache_flag($key);
            }
        }
        return $func($module, $key, $add_user);
    }
    jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
    return false;
}

/**
 * Cache: Cache a string for a given key
 * @param string $module Module doing the caching
 * @param string $key Unique key for cache item
 * @param mixed $value Value to cache
 * @param int $expire_seconds How long key will be cached for (in seconds)
 * @param int $profile_id Profile ID cache item belongs to
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to set user_id to 0
 * @param bool $add_skin DEPRECATED Add active skin to unique cache key
 * @param bool $add_user_info Set to FALSE to not add user_id, language and device type
 * @return mixed returns string on success, bool false on not cached
 */
function jrCore_add_to_cache($module, $key, $value, $expire_seconds = 0, $profile_id = 0, $add_user = true, $add_skin = true, $add_user_info = true)
{
    if (!jrCore_caching_is_enabled()) {
        return true;
    }
    $temp = jrCore_get_active_cache_system();
    $func = "_{$temp}_add_to_cache";
    if (function_exists($func)) {
        $uniq = null;
        // @note: the datastore_cache_profile_ids flag is set in jrCore_db_search_items
        // when generating a LIST of items - this way if one of the items in the list
        // is UPDATED we can reset the cache for the listing
        if ($_ci = jrCore_get_flag('datastore_cache_profile_ids')) {
            if (count($_ci) === 1) {
                $pid = reset($_ci);
                if ($pid != $profile_id) {
                    if ($profile_id === 0) {
                        $profile_id = $pid;
                    }
                    else {
                        $uniq = $pid;
                    }
                }
            }
            else {
                $uniq = implode(',', $_ci);
            }
            jrCore_delete_flag('datastore_cache_profile_ids');
            unset($_ci);
        }
        $uid = ($add_user && jrUser_is_logged_in()) ? intval(jrCore_get_user_session_key('_user_id')) : 0;
        $val = jrCore_strip_emoji($value);
        $key = jrCore_get_cache_key($module, $key, $uid, $add_user_info);
        jrCore_set_cache_flag($key, array((time() + $expire_seconds), $val));
        return $func($module, $key, $val, $expire_seconds, $profile_id, $add_user, $uniq);
    }
    jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
    return false;
}

/**
 * Generate an Internal MD5 Cache key from params
 * @param string $module Module cache key is being created for
 * @param string $key Key to create new key from
 * @param int $user_id User ID (0 for not logged in)
 * @param bool $add_user_info Set to FALSE to not add user_id, language and device type
 * @return string
 */
function jrCore_get_cache_key($module, $key, $user_id = 0, $add_user_info = true)
{
    $key = jrCore_get_base_url() . $module . trim($key) . jrCore_get_config_value('jrCore', 'active_skin', '');
    if ($add_user_info) {
        $key .= ($user_id >= 0) ? intval($user_id) : 0;
        if ($tmp = jrCore_get_user_session_key('user_language')) {
            $key .= $tmp;
        }
        else {
            $key .= 'en-US';
        }
        $dvc = 'desktop';
        if (jrCore_is_tablet_device()) {
            $dvc = 'tablet';
        }
        elseif (jrCore_is_mobile_device()) {
            $dvc = 'mobile';
        }
        $key .= $dvc;
    }
    $key = jrCore_trigger_event('jrCore', 'get_cache_key', $key, func_get_args());
    return md5($key);
}

/**
 * Perform maintenance on the cache system
 * @note This is called ONCE (per cluster) per minute during minute maintenance
 * @return bool
 */
function jrCore_cache_maintenance()
{
    if (!jrCore_caching_is_enabled()) {
        return true;
    }
    // Are we running the core MySQL Cache system?
    $temp = jrCore_get_active_cache_system();
    if ($temp != 'jrCore_mysql' && jrCore_local_cache_is_enabled()) {
        $minute = date('i');
        // We are NOT using MySQL caching at all - make sure core cache table is kept clean every 10 minutes
        if (strpos(" {$minute}", '7')) {
            _jrCore_mysql_delete_all_cache_entries();
        }
    }
    $func = "_{$temp}_cache_maintenance";
    if (function_exists($func)) {
        return $func();
    }
    jrCore_logger('CRI', "core: active cache system function: {$func} is not defined");
    return false;
}

/**
 * Set a new temp global cache flag
 * @param string $flag Unique flag string to set value for
 * @param mixed $value Value to store
 * @return bool
 */
function jrCore_set_cache_flag($flag, $value)
{
    $GLOBALS['__JR_CACHE'][$flag] = $value;
    return true;
}

/**
 * Retrieve a previously set temp global cache flag
 * @param mixed $flag String or Array to save to flag
 * @return mixed
 */
function jrCore_get_cache_flag($flag)
{
    return (isset($GLOBALS['__JR_CACHE'][$flag])) ? $GLOBALS['__JR_CACHE'][$flag] : false;
}

/**
 * delete a previously set temp global cache flag
 * @param mixed $flag String or Array to delete
 * @return bool
 */
function jrCore_delete_cache_flag($flag)
{
    if (isset($GLOBALS['__JR_CACHE'][$flag])) {
        unset($GLOBALS['__JR_CACHE'][$flag]);
        return true;
    }
    return false;
}

/**
 * Delete all set global cache flags
 * @return bool
 */
function jrCore_delete_all_cache_flags()
{
    unset($GLOBALS['__JR_CACHE']);
    return true;
}

//-------------------------------------------
// MySQL Cache Plugins
//-------------------------------------------

/**
 * Internal jrCore_delete_all_cache_entries() plugin
 * @param string $module Optionally delete all cache entries for specific module
 * @param mixed $user_id int|array Optionally delete all cache entries for a specific user_id or array of user_id's
 * @return bool
 * @ignore
 */
function _jrCore_mysql_delete_all_cache_entries($module = null, $user_id = null)
{
    $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
    $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
    if (is_null($module) && is_null($user_id)) {
        $_rq = array(
            "TRUNCATE TABLE {$tb1}",
            "TRUNCATE TABLE {$tb2}"
        );
        jrCore_db_multi_query($_rq, false);
    }
    else {
        $req = "DELETE p, v FROM {$tb2} p LEFT JOIN {$tb1} v ON (v.cache_key = p.cache_key) ";
        if (!is_null($module)) {
            $req .= "WHERE v.cache_module = '" . jrCore_db_escape($module) . "'";
            if (!is_null($user_id)) {
                if (is_array($user_id)) {
                    $req .= " AND v.cache_user_id IN(" . implode(',', $user_id) . ')';
                }
                else {
                    $req .= " AND v.cache_user_id = '" . intval($user_id) . "'";
                }
            }
        }
        else {
            if (is_array($user_id)) {
                $req .= "WHERE v.cache_user_id IN(" . implode(',', $user_id) . ')';
            }
            else {
                $req .= "WHERE v.cache_user_id = '" . intval($user_id) . "'";
            }
        }
        jrCore_db_query($req, null, false, null, false);
    }
    return true;
}

/**
 * Core MySQL cache function to reset cache entries for specified profile IDs
 * @note This is called just before sending data to the user
 * @param $_profile_ids array Array of profile_id => module entries to delete
 * @return bool
 * @ignore
 */
function _jrCore_mysql_process_exit_delete_profile_cache($_profile_ids)
{
    $_in = array();
    foreach ($_profile_ids as $pid => $mod) {
        $pid = (int) $pid;
        if ($mod != 0) {
            // This is a profile reset for a specific module
            $_in[] = "(p.cache_profile_id = {$pid} AND v.cache_module = '{$mod}')";
        }
        else {
            $_in[] = "p.cache_profile_id = {$pid}";
        }
    }
    if (count($_in) > 0) {
        $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
        $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
        $req = "DELETE p, v FROM {$tb2} p LEFT JOIN {$tb1} v ON (v.cache_key = p.cache_key) WHERE " . implode(' OR ', $_in);
        jrCore_db_query($req, null, false, null, false);
    }
    return true;
}

/**
 * Cache: Delete cache for a given key
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @return bool
 */
function _jrCore_mysql_delete_cache($module, $key)
{
    $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
    $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
    $req = "DELETE p, v FROM {$tb2} p LEFT JOIN {$tb1} v ON (v.cache_key = p.cache_key) WHERE p.cache_key = '" . jrCore_db_escape($key) . "'";
    jrCore_db_query($req, null, false, null, false);
    return true;
}

/**
 * Cache: Delete cache for multiple module/keys
 * @param array $_items Cache Items to delete
 * @return bool
 */
function _jrCore_mysql_delete_multiple_cache_entries($_items)
{
    if (is_array($_items)) {
        $_ky = array();
        foreach ($_items as $v) {
            $_ky[] = "p.cache_key = '" . jrCore_db_escape($v[1]) . "'";
        }
        $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
        $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
        $req = "DELETE p, v FROM {$tb2} p LEFT JOIN {$tb1} v ON (v.cache_key = p.cache_key) WHERE " . implode(' OR ', $_ky);
        jrCore_db_query($req, null, false, null, false);
        return true;
    }
    return false;
}

/**
 * Cache: Delete expired cache entries from the cache table
 * @note This runs as a worker post response
 */
function _jrCore_mysql_cache_maintenance()
{
    // On slower disk based systems this query can be a bit slower
    // but we are in process_exit so no need to complain about it
    // @note: runs in process exit
    jrCore_db_set_slow_query_length(100);
    $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
    $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
    // @note: You cannot use LIMIT on a multiple table delete query
    $req = "DELETE p, v FROM {$tb2} p LEFT JOIN {$tb1} v ON (v.cache_key = p.cache_key) WHERE v.cache_expires < (UNIX_TIMESTAMP() - 30)";
    jrCore_db_query($req, null, false, null, false);
    jrCore_db_restore_slow_query_length();
    return true;
}

/**
 * Cache: Check if a given key is cached
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to override
 * @return mixed returns string on success, bool false on not cached
 */
function _jrCore_mysql_is_cached($module, $key, $add_user = true)
{
    $key = jrCore_db_escape($key);
    $tbl = jrCore_db_table_name('jrCore', 'cache_value');
    $req = "SELECT UNIX_TIMESTAMP() AS db_time, cache_expires, cache_encoded, cache_value FROM {$tbl} WHERE cache_key = '{$key}'";
    // @note: We SKIP_TRIGGERS here since this can cause a mysql_query_init event loop
    $_rt = jrCore_db_query($req, 'SINGLE', false, null, false, null, false, true);

    if ($_rt && isset($_rt['cache_value'])) {
        // See if we have expired...
        if ($_rt['cache_expires'] < $_rt['db_time']) {
            // Update so we avoid a stampede and return false so we rebuild in this process
            $req = "UPDATE {$tbl} SET cache_expires = (cache_expires + 30) WHERE cache_key = '{$key}'";
            jrCore_db_query($req, null, false, null, false, null, false);
            return false;
        }
        switch ($_rt['cache_encoded']) {
            case '1':
                // Array
                $_rt['cache_value'] = json_decode($_rt['cache_value'], true);
                break;
            case '2':
                // Object
                $_rt['cache_value'] = json_decode($_rt['cache_value']);
                break;
        }
        return $_rt['cache_value'];
    }
    return false;
}

/**
 * Cache: Cache a string for a given key
 * @param string $module Module doing the caching
 * @param string $key Unique key for cache item
 * @param mixed $value Value to cache
 * @param int $expire_seconds How long key will be cached for (in seconds)
 * @param int $profile_id Profile ID cache item belongs to
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to override
 * @param string $unique Unique Module-Item_IDs (set in DataStore)
 * @return true
 */
function _jrCore_mysql_add_to_cache($module, $key, $value, $expire_seconds = 0, $profile_id = 0, $add_user = true, $unique = null)
{
    global $_post;
    if (!$expire_seconds) {
        $expire_seconds = jrCore_get_config_value('jrCore', 'default_cache_seconds', 300);
    }
    $expire_seconds = intval($expire_seconds);
    if ($expire_seconds === 0) {
        return true;
    }

    // Check if we are encoding this in the DB
    $enc = 0;
    if (is_array($value)) {
        $value = json_encode($value);
        $enc   = 1;
    }
    elseif (is_object($value)) {
        $value = json_encode($value);
        $enc   = 2;
    }

    $pid = 0;
    if ($profile_id == 0) {
        if ($tmp = jrCore_get_flag('jrprofile_view_is_active')) {
            $pid = $tmp;
        }
        elseif (isset($_post['_profile_id'])) {
            $pid = (int) $_post['_profile_id'];
        }
    }
    else {
        $pid = (int) $profile_id;
    }
    if (!$uid = jrCore_get_user_session_key('_user_id')) {
        $uid = 0;
    }
    $val = jrCore_db_escape($value);

    // Start queries
    $tb1 = jrCore_db_table_name('jrCore', 'cache_value');
    $tb2 = jrCore_db_table_name('jrCore', 'cache_profile_id');
    $_rq = array(
        "INSERT INTO {$tb1} (cache_key, cache_expires, cache_module, cache_user_id, cache_encoded, cache_value)
         VALUES ('{$key}', (UNIX_TIMESTAMP() + {$expire_seconds}), '{$module}', {$uid}, {$enc}, '{$val}')
         ON DUPLICATE KEY UPDATE cache_expires = VALUES(cache_expires), cache_encoded = {$enc}, cache_value = VALUES(cache_value)"
    );
    // @note $unq will be a comma separated list of PROFILE_IDS only - not DS item_ids
    $_un = array();
    if (!is_null($unique)) {
        foreach (explode(',', $unique) as $p) {
            $p       = (int) $p;
            $_un[$p] = "('{$key}',{$p})";
        }
    }
    $_un[$pid] = "('{$key}',{$pid})";
    $_rq[]     = "INSERT IGNORE INTO {$tb2} (cache_key, cache_profile_id) VALUES " . implode(',', $_un);
    jrCore_db_multi_query($_rq, false, false);
    return true;
}
