<?php
/**
 *
 * This file is part of Aura for PHP.
 *
 * @license http://opensource.org/licenses/bsd-license.php BSD
 *
 */
namespace Aura\Auth\Adapter;

use Aura\Auth\Exception;
use Aura\Auth\Verifier\VerifierInterface;

/**
 *
 * Authenticate against a file generated by htpassword.
 *
 * Format for each line is "username:hashedpassword\n";
 *
 * Automatically checks against DES, SHA, and apr1-MD5.
 *
 * SECURITY NOTE: Default DES encryption will only check up to the first
 * 8 characters of a password; chars after 8 are ignored.  This means
 * that if the real password is "atechars", the word "atecharsnine" would
 * be valid.  This is bad.  As a workaround, if the password provided by
 * the user is longer than 8 characters, and DES encryption is being
 * used, this class will *not* validate it.
 *
 * @package Aura.Auth
 *
 */
class HtpasswdAdapter extends AbstractAdapter
{
    /**
     *
     * The httpasswd credential file.
     *
     * @var string
     *
     */
    protected $file;

    /**
     *
     * A verifier for passwords.
     *
     * @var VerifierInterface
     *
     */
    protected $verifier;

    /**
     *
     * Constructor.
     *
     * @param string $file The htpasswd file path.
     *
     * @param VerifierInterface $verifier
     *
     * @return null
     */
    public function __construct($file, VerifierInterface $verifier)
    {
        $this->file = $file;
        $this->verifier = $verifier;
    }

    /**
     *
     * Return object of type VerifierInterface
     *
     * @return VerifierInterface
     *
     */
    public function getVerifier()
    {
        return $this->verifier;
    }

    /**
     *
     * Verifies a set of credentials.
     *
     * @param array $input An array with keys 'username' and 'password'.
     *
     * @return array An array of login data.
     *
     */
    public function login(array $input)
    {
        $this->checkInput($input);
        $username = $input['username'];
        $password = $input['password'];
        $hashvalue = $this->fetchHashedPassword($username);
        $this->verify($password, $hashvalue);
        return array($username, array());
    }

    /**
     *
     * Reads the hashed password for a username from the htpasswd file.
     *
     * @param string $username The username to find in the htpasswd file.
     *
     * @return string
     *
     * @throws Exception\UsernameNotFound when the username is not found in the
     * htpasswd file.
     *
     */
    protected function fetchHashedPassword($username)
    {
        // force the full, real path to the file
        $real = realpath($this->file);
        if (! $real) {
            throw new Exception\FileNotReadable($this->file);
        }

        // find the user's line in the file
        $fp = fopen($real, 'r');
        $len = strlen($username) + 1;
        $hashvalue = false;
        while ($line = fgets($fp)) {
            if (substr($line, 0, $len) == "{$username}:") {
                // found the line, leave the loop
                $tmp = explode(':', trim($line));
                $hashvalue = $tmp[1];
                break;
            }
        }

        // close the file
        fclose($fp);

        // did we find the encrypted password for the username?
        if ($hashvalue) {
            return $hashvalue;
        }

        throw new Exception\UsernameNotFound;
    }

    /**
     *
     * Verifies the input password against the hashed password.
     *
     * @param string $password The input password.
     *
     * @param string $hashvalue The hashed password in htpasswd.
     *
     * @return null
     *
     * @throws Exception\PasswordIncorrect on failed verification.
     *
     */
    protected function verify($password, $hashvalue)
    {
        if (! $this->verifier->verify($password, $hashvalue)) {
            throw new Exception\PasswordIncorrect;
        }
    }
}
