<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Classe pour la génération de liens d'action dans Flatboard.
 *
 * Fournit des méthodes statiques pour créer des liens HTML avec icônes et info-bulles,
 * utilisés pour gérer les topics, réponses, forums, plugins, bans, tags et modérateurs.
 */
class EntryLink
{
    /** @var string Classe CSS par défaut pour les badges */
    private const DEFAULT_CLASS = 'primary';

    /** @var string Espace par défaut après un lien */
    private const DEFAULT_SPACE = ' ';

    /** @var string Direction par défaut de l'info-bulle */
    private const DEFAULT_TOOLTIP_DIR = 'top';

    /** @var string Séparateur de chemin */
    private const DS = DIRECTORY_SEPARATOR;

    /** @var array Configuration des types de liens */
    private const LINK_TYPES = [
        'topic' => [
            'edit' => ['path' => 'edit.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit'],
            'delete' => ['path' => 'delete.php', 'icon' => 'fa fa-trash-o', 'hint' => 'delete', 'class' => 'danger']
        ],
        'reply' => [
            'edit' => ['path' => 'edit.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit'],
            'delete' => ['path' => 'delete.php', 'icon' => 'fa fa-trash-o', 'hint' => 'delete', 'class' => 'danger']
        ],
        'forum' => [
            'edit' => ['path' => 'edit.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit'],
            'delete' => ['path' => 'delete.php', 'icon' => 'fa fa-trash-o', 'hint' => 'delete', 'class' => 'danger']
        ],
        'plugin' => [
            'edit' => ['path' => 'config.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit']
        ],
        'ban' => [
            'ban' => ['path' => 'add.php', 'icon' => 'fa fa-ban', 'hint' => 'ban_user'],
            'unban' => ['path' => 'delete.php', 'icon' => 'fa fa-circle-o', 'hint' => 'unban_user', 'class' => 'danger']
        ],
        'tag' => [
            'edit' => ['path' => 'edit.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit'],
            'delete' => ['path' => 'delete.php', 'icon' => 'fa fa-trash-o', 'hint' => 'delete', 'class' => 'danger']
        ],
        'worker' => [
            'edit' => ['path' => 'edit.php', 'icon' => 'fa fa-pencil-square-o', 'hint' => 'edit']
        ]
    ];

    /**
     * Constructeur protégé pour empêcher l'instanciation.
     */
    protected function __construct()
    {
        // Classe statique
    }

    /**
     * Crée un lien HTML avec icône et info-bulle.
     *
     * @param string $link URL du lien.
     * @param string $hint Clé de traduction pour l'info-bulle.
     * @param string $icon Classe de l'icône FontAwesome.
     * @param string $class Classe CSS du badge (par défaut : 'primary').
     * @param string $space Espace après le lien (par défaut : ' ').
     * @param string $tooltipDir Direction de l'info-bulle (par défaut : 'top').
     * @param array $lang Tableau des traductions (facultatif, utilise global $lang si vide).
     * @return string HTML du lien généré.
     */
    public static function createLink(
        string $link,
        string $hint,
        string $icon,
        string $class = self::DEFAULT_CLASS,
        string $space = self::DEFAULT_SPACE,
        string $tooltipDir = self::DEFAULT_TOOLTIP_DIR,
        array $lang = []
    ): string {
        global $lang;
        // Utilise le tableau $lang passé en paramètre ou le global $lang
        $hintText = isset($lang[$hint]) ? $lang[$hint] : ($lang[$hint] ?? $hint);

        return sprintf(
            '<a href="%s" class="badge badge-%s" data-toggle="tooltip" data-placement="%s" title="%s"><i class="%s"></i></a>%s',
            htmlspecialchars($link, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($class, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($tooltipDir, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($hintText, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($icon, ENT_QUOTES, 'UTF-8'),
            $space
        );
    }

    /**
     * Génère des liens d'action pour une entité spécifique.
     *
     * @param string $type Type d'entité ('topic', 'reply', 'forum', 'plugin', 'ban', 'tag', 'worker').
     * @param string $id Identifiant de l'entité.
     * @param callable $permissionCheck Fonction de vérification des permissions.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens d'action.
     */
    private static function manageEntity(string $type, string $id, callable $permissionCheck, array $lang = []): string
    {
        if (!$permissionCheck()) {
            return Plugin::hook("manage" . ucfirst($type), $id);
        }

        $links = '';
        foreach (self::LINK_TYPES[$type] as $action => $config) {
            $link = sprintf('%s%s%s%s%s', $config['path'], self::DS, $type, self::DS, $id);
            $links .= self::createLink(
                link: $link,
                hint: $config['hint'],
                icon: $config['icon'],
                class: $config['class'] ?? self::DEFAULT_CLASS,
                lang: $lang
            );
        }

        return $links . Plugin::hook("manage" . ucfirst($type), $id);
    }

    /**
     * Génère des liens pour gérer un topic.
     *
     * @param string $topic ID du topic.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function manageTopic(string $topic, array $lang = []): string
    {
        return self::manageEntity('topic', $topic, fn() => User::isWorker() || User::isAuthor($topic, 'topic'), $lang);
    }

    /**
     * Génère des liens pour gérer une réponse.
     *
     * @param string $reply ID de la réponse.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function manageReply(string $reply, array $lang = []): string
    {
        return self::manageEntity('reply', $reply, fn() => User::isWorker() || User::isAuthor($reply, 'reply'), $lang);
    }

    /**
     * Génère des liens pour gérer un forum.
     *
     * @param string $forum ID du forum.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function manageForum(string $forum, array $lang = []): string
    {
        return self::manageEntity('forum', $forum, fn() => User::isAdmin(), $lang);
    }

    /**
     * Génère des liens pour gérer un plugin.
     *
     * @param string $plugin ID du plugin.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function managePlugin(string $plugin, array $lang = []): string
    {
        return self::manageEntity('plugin', $plugin, fn() => User::isAdmin(), $lang);
    }

    /**
     * Génère des liens pour bannir ou débannir un utilisateur.
     *
     * @param string $user ID de l'utilisateur.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function userBan(string $user, array $lang = []): string
    {
        return self::manageEntity('ban', $user, fn() => User::isAdmin(), $lang);
    }

    /**
     * Génère des liens pour gérer un tag.
     *
     * @param string $tag ID du tag.
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function manageTag(string $tag, array $lang = []): string
    {
        return self::manageEntity('tag', $tag, fn() => User::isAdmin(), $lang);
    }

    /**
     * Génère des liens pour gérer un modérateur.
     *
     * @param string $password Identifiant du modérateur.
     * @param array $config Configuration des modérateurs (facultatif).
     * @param array $lang Tableau des traductions (facultatif).
     * @return string HTML des liens.
     */
    public static function manageWorker(string $password, array $config = [], array $lang = []): string
    {
        global $config;
        $config = $config ?: $config;

        return self::manageEntity(
            'worker',
            $password,
            fn() => isset($_SESSION['worker']) && isset($config['worker'][$password]) && $_SESSION['worker'] === $config['worker'][$password],
            $lang
        );
    }
}