<?php
/*
 * Nom du projet : Flatboard
 * URL du projet : https://flatboard.org
 * Auteur : Frédéric Kaplon et contributeurs
 * Tout le code de Flatboard est publié sous la licence MIT.
 */

// Définition du type de contenu pour la sortie
$out['self'] = 'feed';
require_once __DIR__ . '/header.php';

// Configuration des en-têtes HTTP
header('Content-Type: application/atom+xml; charset=utf-8');
header('Cache-Control: public, max-age=300'); // Cache de 5 minutes

// Constantes pour améliorer la lisibilité
const FEED_NAMESPACE = 'http://www.w3.org/2005/Atom';
const XML_ENCODING = 'UTF-8';
const DEFAULT_ENTRIES_LIMIT = 20;

/**
 * Classe utilitaire pour la génération de flux RSS/Atom
 */
class AtomFeedGenerator
{
    private array $config;
    private array $out;
    private array $lang;

    public function __construct(array $config, array $out, array $lang)
    {
        $this->config = $config;
        $this->out = $out;
        $this->lang = $lang;
    }

    /**
     * Échappe et formate du contenu pour XML
     */
    private function escapeXml(string $content): string
    {
        return htmlspecialchars($content, ENT_XML1 | ENT_QUOTES, XML_ENCODING);
    }

    /**
     * Génère une entrée Atom
     */
    private function generateAtomEntry(string $id, string $title, string $url, string $updated, string $summary): string
    {
        $escapedId = $this->escapeXml($id);
        $escapedTitle = $this->escapeXml($title);
        $escapedUrl = $this->escapeXml($url);
        $escapedSummary = $this->escapeXml($summary);

        return <<<XML
    <entry>
        <id>{$escapedId}</id>
        <title>{$escapedTitle}</title>
        <updated>{$updated}</updated>
        <link href="{$escapedUrl}"/>
        <summary type="html">{$escapedSummary}</summary>
    </entry>
XML;
    }

    /**
     * Génère l'en-tête du flux Atom
     */
    private function generateFeedHeader(): string
    {
        $title = $this->escapeXml($this->config['name']);
        $baseUrl = $this->escapeXml($this->out['baseURL']);
        $updated = date('c');

        return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
    <title>{$title}</title>
    <link href="{$baseUrl}"/>
    <updated>{$updated}</updated>
    <author><name>{$title}</name></author>
    <id>{$baseUrl}</id>
XML;
    }

    /**
     * Traite et génère les entrées pour les sujets
     */
    private function processTopicEntries(array $topics, string $type = 'topic', int $limit = DEFAULT_ENTRIES_LIMIT): string
    {
        if (empty($topics)) {
            return $this->generateEmptyEntry("No {$type}s found");
        }

        // Tri et limitation des résultats
        arsort($topics, SORT_NATURAL | SORT_FLAG_CASE);
        $topics = array_slice($topics, 0, $limit, true);

        $entries = [];
        foreach ($topics as $topicId) {
            $entry = $this->generateTopicEntry($topicId, $type);
            if ($entry) {
                $entries[] = $entry;
            }
        }

        return implode("\n", $entries);
    }

    /**
     * Génère une entrée pour un sujet spécifique
     */
    private function generateTopicEntry(string $topicId, string $type): ?string
    {
        $topicEntry = flatDB::readEntry($type, $topicId);
        if (!$topicEntry) {
            return null;
        }

        $entryData = $this->buildEntryData($topicId, $topicEntry, $type);
        if (!$entryData) {
            return null;
        }

        return $this->generateAtomEntry(
            $entryData['id'],
            $entryData['title'],
            $entryData['url'],
            $entryData['updated'],
            $entryData['summary']
        );
    }

    /**
     * Construit les données d'une entrée
     */
    private function buildEntryData(string $topicId, array $topicEntry, string $type): ?array
    {
        $baseData = [
            'id' => $this->out['baseURL'] . 'view.php' . DS . $type . DS . $topicId,
            'updated' => Util::toDate($topicId, 'c'),
            'summary' => Parser::summary(Parser::content($topicEntry['content'], true))
        ];

        if ($type === 'reply') {
            return $this->buildReplyEntryData($topicId, $topicEntry, $baseData);
        }

        return array_merge($baseData, [
            'title' => $topicEntry['trip'] . ': ' . $topicEntry['title'],
            'url' => 'view.php' . DS . $type . DS . $topicId
        ]);
    }

    /**
     * Construit les données spécifiques pour une réponse
     */
    private function buildReplyEntryData(string $topicId, array $topicEntry, array $baseData): ?array
    {
        $forumTopic = flatDB::readEntry('topic', $topicEntry['topic']);
        if (!$forumTopic) {
            return null;
        }

        $pageNumber = Util::onPage($topicId, $forumTopic['reply']);
        $url = 'view.php' . DS . 'topic' . DS . $topicEntry['topic'] . 
               DS . 'p' . DS . $pageNumber . '#' . $topicId;

        return array_merge($baseData, [
            'title' => $topicEntry['trip'] . ': ' . $forumTopic['title'],
            'url' => $url
        ]);
    }

    /**
     * Génère une entrée vide avec un message
     */
    private function generateEmptyEntry(string $message): string
    {
        return $this->generateAtomEntry(
            $this->out['baseURL'],
            $message,
            '',
            date('c'),
            $message
        );
    }

    /**
     * Génère le flux pour tous les sujets
     */
    public function generateTopicsFeed(): string
    {
        $this->out['subtitle'] = $this->lang['topic'];
        $this->out['type'] = 'topic';
        
        $topics = flatDB::listEntry('topic');
        return $this->processTopicEntries($topics, 'topic');
    }

    /**
     * Génère le flux pour toutes les réponses
     */
    public function generateRepliesFeed(): string
    {
        $this->out['subtitle'] = $this->lang['reply'];
        $this->out['type'] = 'reply';
        
        $replies = flatDB::listEntry('reply');
        return $this->processTopicEntries($replies, 'reply');
    }

    /**
     * Génère le flux pour un forum spécifique
     */
    public function generateForumFeed(string $forumId): string
    {
        $forumEntry = flatDB::readEntry('forum', $forumId);
        if (!$forumEntry) {
            return $this->generateEmptyEntry('Forum not found');
        }

        $this->out['subtitle'] = $this->escapeXml($forumEntry['name']);
        $this->out['type'] = 'forum';
        
        $topics = $forumEntry['topic'] ?? [];
        return $this->processTopicEntries($topics, 'topic');
    }

    /**
     * Génère le flux pour un sujet spécifique
     */
    public function generateTopicFeed(string $topicId): string
    {
        $topicEntry = flatDB::readEntry('topic', $topicId);
        if (!$topicEntry) {
            return $this->generateEmptyEntry('Topic not found');
        }

        $this->out['subtitle'] = $this->escapeXml($topicEntry['title']);
        $this->out['type'] = 'thread';

        $url = 'view.php' . DS . 'topic' . DS . $topicId;
        return $this->generateAtomEntry(
            $this->out['baseURL'] . $url,
            $topicEntry['trip'] . ': ' . $topicEntry['title'],
            $url,
            Util::toDate($topicId, 'c'),
            Parser::summary(Parser::content($topicEntry['content'], true))
        );
    }

    /**
     * Génère le flux complet
     */
    public function generateFeed(): string
    {
        $content = '';

        // Détermine le type de flux à générer
        if (Util::isGET('topic')) {
            $content = $this->generateTopicsFeed();
        } elseif (Util::isGET('reply')) {
            $content = $this->generateRepliesFeed();
        } elseif (Util::isGETValidEntry('forum', 'forum')) {
            $content = $this->generateForumFeed($_GET['forum']);
        } elseif (Util::isGETValidEntry('topic', 'topic')) {
            $content = $this->generateTopicFeed($_GET['topic']);
        } else {
            Util::redirect('index.php' . DS . '404');
            return '';
        }

        return $this->generateFeedHeader() . "\n" . $content . "\n</feed>";
    }
}

// Utilisation de la classe optimisée
try {
    $feedGenerator = new AtomFeedGenerator($config, $out, $lang);
    echo $feedGenerator->generateFeed();
} catch (Exception $e) {
    // Log de l'erreur et redirection vers la page d'erreur
    error_log("RSS Feed Error: " . $e->getMessage());
    Util::redirect('index.php' . DS . '404');
}
?>