<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Classe pour la gestion des utilisateurs dans Flatboard.
 *
 * Fournit des méthodes statiques pour l'authentification, la vérification des rôles,
 * la gestion des clés, la protection des données et la gestion des bannissements.
 *
 */
class User
{
    /** @var string Chemin du répertoire des données */
    private static $DATA_DIR = DATA_DIR;

    /** @var string Nom du fichier de clé */
    private static $KEY_FILE = 'key.php';

    /** @var string Nom du fichier de liste de bannissement */
    private static $BAN_FILE = BAN_FILE;

    /** @var string Rôle administrateur */
    private static $ROLE_ADMIN = 'admin';

    /** @var string Rôle modérateur */
    private static $ROLE_WORKER = 'worker';

    /** @var string Méthode de chiffrement */
    private static $ENCRYPT_METHOD = 'AES-256-CBC';

    /**
     * Constructeur protégé pour empêcher l'instanciation.
     */
    protected function __construct()
    {
        // Classe statique
    }

    /**
     * Vérifie si l'utilisateur a des droits administratifs.
     *
     * @return bool True si l'utilisateur est administrateur, false sinon.
     */
    public static function isAdmin()
    {
        return isset($_SESSION['role']) && $_SESSION['role'] === self::$ROLE_ADMIN;
    }

    /**
     * Vérifie si l'utilisateur a des droits de modérateur ou administrateur.
     *
     * @return bool True si l'utilisateur est modérateur ou administrateur, false sinon.
     */
    public static function isWorker()
    {
        return isset($_SESSION['role']) && in_array($_SESSION['role'], array(self::$ROLE_WORKER, self::$ROLE_ADMIN), true);
    }

    /**
     * Vérifie si l'utilisateur est l'auteur d'une entrée.
     *
     * @param string $entry Identifiant de l'entrée.
     * @param string $data Type d'entrée ('topic', 'reply' ou autre).
     * @param string|null $sessionTrip Identifiant de session (par défaut : global $sessionTrip).
     * @return bool True si l'utilisateur est l'auteur, false sinon.
     * @throws RuntimeException Si l'entrée ne peut pas être lue.
     */
    public static function isAuthor($entry, $data = '', $sessionTrip = null)
    {
        global $sessionTrip;
        $sessionTrip = isset($sessionTrip) ? $sessionTrip : $sessionTrip;

        if (empty($entry)) {
            return false;
        }

        if (in_array($data, array('topic', 'reply'), true)) {
            $entryData = flatDB::readEntry($data, $entry);
            if ($entryData === null) {
                throw new RuntimeException("Impossible de lire l'entrée '$entry' de type '$data'.");
            }

            if (!isset($sessionTrip)) {
                return false;
            }

            $tripCrypt = HTMLForm::trip($sessionTrip, $entry);
            $isAuthor = $tripCrypt === (isset($entryData['trip']) ? $entryData['trip'] : '');

            return strpos($tripCrypt, '@') !== false ? $isAuthor : isset($_SESSION[$entry]);
        }

        return isset($_SESSION[$entry]);
    }

    /**
     * Authentifie un utilisateur.
     *
     * @param string $trip Identifiant de l'utilisateur.
     * @param array $config Configuration du site (par défaut : global $config).
     * @param array $lang Traductions (par défaut : global $lang).
     * @return bool True si l'authentification réussit, false sinon.
     * @throws InvalidArgumentException Si la syntaxe de l'identifiant est invalide.
     */
    public static function login($trip, $config = array(), $lang = array())
    {
        global $config, $lang;
        $config = $config ? $config : $config;
        $lang = $lang ? $lang : $lang;

        if (empty($trip)) {
            $_SESSION['bad_user_syntax'] = 1;
            return false;
        }

        $tripNocrypt = HTMLForm::clean(Parser::translitIt($trip));
        $tripCrypt = HTMLForm::trip($tripNocrypt, $trip);

        // Authentification administrateur
        if (isset($config['admin']) && $tripCrypt === $config['admin']) {
            $_SESSION['role'] = self::$ROLE_ADMIN;
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = isset($config['mail']) ? $config['mail'] : '';
            return true;
        }

        // Authentification modérateur
        if (isset($config['worker'][$tripCrypt])) {
            $_SESSION['role'] = self::$ROLE_WORKER;
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = $config['worker'][$tripCrypt];
            return true;
        }

        // Authentification utilisateur (format email)
        if (strpos($trip, '@') !== false) {
            $parts = explode('@', $trip);
            if (count($parts) === 2 && strlen($parts[1]) >= 4) {
                $_SESSION['trip'] = $tripNocrypt;
                return true;
            }
            $_SESSION['bad_user_syntax'] = 1;
            return false;
        }

        // Échec de l'authentification
        $_SESSION['bad_user_syntax'] = 1;
        return false;
    }

    /**
     * Regénère le fichier de clé pour l'administrateur.
     *
     * @param string $password Nouveau mot de passe.
     * @param array $config Configuration du site (par défaut : global $config).
     * @return bool True si la clé est générée avec succès, false sinon.
     * @throws RuntimeException Si le fichier de clé ne peut pas être écrit.
     */
    public static function generateKey($password, $config = array())
    {
        global $config;
        $config = $config ? $config : $config;

        if (empty($password)) {
            return false;
        }

        $keyFile = self::$DATA_DIR . self::$KEY_FILE;

        // Supprime l'ancien fichier de clé
        if (file_exists($keyFile) && !unlink($keyFile)) {
            throw new RuntimeException("Impossible de supprimer l'ancien fichier de clé '$keyFile'.");
        }

        // Génère une nouvelle clé
        $key = password_hash($password, PASSWORD_DEFAULT);
        $keyContent = "<?php define('KEY', " . var_export($key, true) . "); ?>";
        if (file_put_contents($keyFile, $keyContent, LOCK_EX) === false) {
            throw new RuntimeException("Impossible d'écrire le fichier de clé '$keyFile'.");
        }

        // Met à jour la configuration avec le nouveau hash admin
        $config['admin'] = hash_hmac('sha1', $password, defined('KEY') ? KEY : $key);
        if (!flatDB::saveEntry('config', 'config', $config)) {
            throw new RuntimeException("Impossible de sauvegarder la configuration avec le nouveau hash admin.");
        }

        $_SESSION['role'] = '';
        return true;
    }

    /**
     * Chiffre ou déchiffre une chaîne avec AES-256-CBC.
     *
     * @param string $string Chaîne à chiffrer/déchiffrer.
     * @param string $action 'e' pour chiffrer, 'd' pour déchiffrer.
     * @return string|bool Chaîne chiffrée ou déchiffrée, false en cas d'erreur.
     * @throws RuntimeException Si la clé n'est pas définie.
     */
    public static function simple_crypt($string, $action = 'e')
    {
        if (!defined('KEY')) {
            throw new RuntimeException("La constante KEY n'est pas définie pour le chiffrement.");
        }

        $key = hash('sha256', KEY);
        $iv = substr(hash('sha256', KEY), 0, 16);

        if ($action === 'e') {
            return base64_encode(openssl_encrypt($string, self::$ENCRYPT_METHOD, $key, 0, $iv));
        }

        if ($action === 'd') {
            return openssl_decrypt(base64_decode($string), self::$ENCRYPT_METHOD, $key, 0, $iv);
        }

        return false;
    }

    /**
     * Masque un mot de passe en remplaçant les caractères internes par des étoiles.
     *
     * @param string $password Mot de passe à masquer.
     * @return string Mot de passe masqué.
     */
    public static function getStarred($password)
    {
        $len = strlen($password);
        if ($len < 2) {
            return $password;
        }
        return substr($password, 0, 1) . str_repeat('*', $len - 2) . substr($password, -1);
    }

    /**
     * Protège un email en générant un lien mailto via JavaScript.
     *
     * @param string $email Adresse email à protéger.
     * @param string $word Texte à afficher pour le lien.
     * @return string Code HTML/JavaScript pour le lien email.
     * @throws InvalidArgumentException Si l'email est invalide.
     */
    public static function protect_email($email, $word)
    {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new InvalidArgumentException("Adresse email invalide : '$email'.");
        }

        $parts = explode('@', $email);
        $jsCode = sprintf(
            '<script>var a="<a href=\'mailto:";var b="%s";var c="%s";var d="\' class=\'badge badge-dark\'><i class=\'fa fa-envelope\'></i> ";var e="%s";var f="</a>";document.write(a+b+"@"+c+d+e+f);</script><noscript>Activer JavaScript pour afficher le mail</noscript>',
            htmlspecialchars($parts[0], ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($parts[1], ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($word, ENT_QUOTES, 'UTF-8')
        );

        return $jsCode;
    }

    /**
     * Récupère l'adresse IP réelle de l'utilisateur.
     *
     * @return string Adresse IP détectée.
     */
    public static function getRealIpAddr()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        }
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
    }

    /**
     * Vérifie si une adresse IP est bannie.
     *
     * @param string $ip Adresse IP à vérifier.
     * @return bool True si l'IP est bannie, false sinon.
     */
    public static function isBan($ip)
    {
        if (empty($ip) || !file_exists(self::$BAN_FILE)) {
            return false;
        }

        $blacklist = file_get_contents(self::$BAN_FILE);
        return $blacklist !== false && strpos($blacklist, $ip . "\n") !== false;
    }

    /**
     * Vérifie si l'IP de l'utilisateur est bannie et redirige si nécessaire.
     *
     * @param array $config Configuration du site (par défaut : global $config).
     * @param array $lang Traductions (par défaut : global $lang).
     * @throws RuntimeException Si le fichier de bannissement ou le thème ne peut pas être chargé.
     */
    public static function checkIP($config = array(), $lang = array())
    {
        global $config, $lang;
        $config = $config ? $config : $config;
        $lang = $lang ? $lang : $lang;

        $ip = self::getRealIpAddr();
        $list = file(self::$BAN_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($list === false) {
            throw new RuntimeException("Impossible de lire le fichier de bannissement '" . self::$BAN_FILE . "'.");
        }

        foreach ($list as $bannedIp) {
            if (strpos($ip, $bannedIp) !== false) {
                $themeFile = THEME_DIR . (isset($config['theme']) ? $config['theme'] : 'default') . DIRECTORY_SEPARATOR . 'banned.tpl.php';
                if (!file_exists($themeFile)) {
                    throw new RuntimeException("Fichier de thème '$themeFile' introuvable pour l'affichage de la page de bannissement.");
                }
                require $themeFile;
                exit;
            }
        }
    }
}