<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
*/

class HTMLForm
{
    // Couleurs CSS valides pour la méthode isValidColor
    private const VALID_COLORS = [
        'aqua', 'black', 'blue', 'fuchsia', 'gray', 'green', 'lime', 'maroon',
        'navy', 'olive', 'purple', 'red', 'silver', 'teal', 'white', 'yellow',
        'aliceblue', 'antiquewhite', 'aquamarine', 'azure', 'beige',
        'bisque', 'blanchedalmond', 'blueviolet', 'brown', 'burlywood',
        'cadetblue', 'chartreuse', 'chocolate', 'coral', 'cornflowerblue',
        'cornsilk', 'crimson', 'cyan', 'darkblue', 'darkcyan', 'darkgoldenrod',
        'darkgray', 'darkgreen', 'darkkhaki', 'darkmagenta', 'darkolivegreen',
        'darkorange', 'darkorchid', 'darkred', 'darksalmon', 'darkseagreen',
        'darkslateblue', 'darkslategray', 'darkturquoise', 'darkviolet',
        'deeppink', 'deepskyblue', 'dimgray', 'dodgerblue', 'firebrick',
        'floralwhite', 'forestgreen', 'gainsboro', 'ghostwhite', 'gold',
        'goldenrod', 'greenyellow', 'honeydew', 'hotpink', 'indianred',
        'indigo', 'ivory', 'khaki', 'lavender', 'lavenderblush', 'lawngreen',
        'lemonchiffon', 'lightblue', 'lightcoral', 'lightcyan',
        'lightgoldenrodyellow', 'lightgray', 'lightgreen', 'lightpink',
        'lightsalmon', 'lightseagreen', 'lightskyblue', 'lightslategray',
        'lightsteelblue', 'lightyellow', 'limegreen', 'linen', 'magenta',
        'mediumaquamarine', 'mediumblue', 'mediumorchid', 'mediumpurple',
        'mediumseagreen', 'mediumslateblue', 'mediumspringgreen',
        'mediumturquoise', 'mediumvioletred', 'midnightblue', 'mintcream',
        'mistyrose', 'moccasin', 'navajowhite', 'oldlace', 'olivedrab',
        'orange', 'orangered', 'orchid', 'palegoldenrod', 'palegreen',
        'paleturquoise', 'palevioletred', 'papayawhip', 'peachpuff', 'peru',
        'pink', 'plum', 'powderblue', 'rosybrown', 'royalblue', 'saddlebrown',
        'salmon', 'sandybrown', 'seagreen', 'seashell', 'sienna', 'skyblue',
        'slateblue', 'slategray', 'snow', 'springgreen', 'steelblue', 'tan',
        'teal', 'thistle', 'tomato', 'turquoise', 'violet', 'wheat',
        'whitesmoke', 'yellow', 'yellowgreen'
    ];

    /**
     * Génère les attributs HTML pour les champs de formulaire.
     *
     * @param string $class Classes CSS supplémentaires pour le champ (facultatif).
     * @param string $placeholder Clé de traduction pour le placeholder (facultatif).
     * @param bool $disabled Indique si le champ est désactivé (par défaut : false).
     * @param string $type Type du champ (par défaut : 'text').
     * @return string Chaîne contenant les attributs HTML.
     */
    private static function getAttributes(string $class = '', string $placeholder = '', bool $disabled = false, string $type = 'text'): string
    {
        global $lang;
        $attrs = [];
        $attrs[] = $class ? 'class="form-control ' . htmlspecialchars($class) . '"' : 'class="form-control"';
  // Ajoute la classe CSS avec ou sans classe supplémentaire
        if ($placeholder && isset($lang[$placeholder])) {
            $attrs[] = 'placeholder="' . htmlspecialchars($lang[$placeholder]) . '"'; // Ajoute un placeholder traduit si disponible
        }
        if ($disabled) {
            $attrs[] = 'disabled'; // Ajoute l'attribut disabled si spécifié
        }
        if ($type) {
            $attrs[] = 'type="' . htmlspecialchars($type) . '"'; // Ajoute le type de champ
        }
        return implode(' ', $attrs); // Retourne les attributs sous forme de chaîne
    }

    /**
     * Génère le HTML pour la description d'un champ de formulaire.
     *
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @return string HTML de la description ou chaîne vide si non défini.
     */
    private static function getDescription(string $desc = ''): string
    {
        global $lang;
        return $desc && isset($lang[$desc]) ? '<small class="form-text text-muted">' . htmlspecialchars($lang[$desc]) . '</small>' : ''; // Retourne la description traduite en petit texte ou rien
    }

    /**
     * Convertit du texte HTML en texte brut sécurisé.
     *
     * @param mixed $text Texte à nettoyer.
     * @return string Texte nettoyé et sécurisé.
     * @throws Exception Si la constante CHARSET n'est pas définie.
     */
    public static function clean($text): string
    {
        if (!defined('CHARSET')) {
            throw new Exception('CHARSET n\'est pas défini.');
        }
        return htmlspecialchars(trim($text ?? ''), ENT_QUOTES, CHARSET); // Nettoie et encode le texte pour éviter les injections XSS
    }

    /**
     * Vérifie si une couleur est valide (format hexadécimal ou nom CSS).
     *
     * @param string $color Couleur à vérifier.
     * @return bool True si la couleur est valide, false sinon.
     */
    public static function isValidColor(string $color): bool
    {
        return preg_match('/^#([0-9A-Fa-f]{3}|[0-9A-Fa-f]{6})$/', $color) || // Vérifie le format hexadécimal (#RGB ou #RRGGBB)
               in_array(strtolower($color), self::VALID_COLORS); // Vérifie si la couleur est dans la liste des noms CSS valides
    }

    /**
     * Normalise les sauts de ligne dans un texte.
     *
     * @param string $text Texte à normaliser.
     * @return string Texte avec des sauts de ligne normalisés.
     */
    public static function transNL(string $text): string
    {
        $text = str_replace(["\r\n", "\r"], "\n", $text); // Remplace tous les types de sauts de ligne par \n
        return preg_replace('/\n{3,}/', "\n\n", $text); // Réduit les sauts de ligne multiples à deux maximum
    }

    /**
     * Génère un tripcode pour l'identification d'un utilisateur.
     *
     * @param string $name Nom de l'utilisateur.
     * @param string $id Identifiant de secours.
     * @return string Tripcode généré.
     */
    public static function trip(string $name, string $id): string
    {
        global $config;
        if (empty($name)) {
            return is_string($id) ? substr($id, -8) : ''; // Retourne les 8 derniers caractères de l'ID si le nom est vide
        }

        $sign = '@';
        [$first, $last] = array_pad(explode($sign, $name, 2), 2, ''); // Sépare le nom en deux parties autour du @
        $first = trim($first);
        $last = trim($last);
        $fallback = $config['salt'] ?? KEY; // Utilise un sel par défaut si non configuré
        $salt = sha1($fallback . md5($last)); // Génère un sel basé sur le dernier élément

        $trip = preg_replace('/[^A-Za-z0-9\-]/', '', str_replace(['http://', 'https://'], '', $first)); // Nettoie le premier élément
        return $trip . ($last ? $sign . substr($salt, -8) : ''); // Combine le premier élément avec une partie du sel si le dernier existe
    }

    /**
     * Affiche un message d'erreur si une erreur est présente en session.
     *
     * @param string $eid Identifiant de l'erreur en session.
     * @param string $msg Message d'erreur à afficher.
     * @return string HTML du message d'erreur ou chaîne vide.
     */
    public static function err(string $eid, string $msg): string
    {
        if (isset($_SESSION[$eid])) {
            unset($_SESSION[$eid]); // Supprime l'erreur de la session après affichage
            return '<span style="color:red; font-weight:500;">' . htmlspecialchars($msg) . '</span>'; // Retourne le message en rouge
        }
        return '';
    }

    /**
     * Génère un champ de mot de passe.
     *
     * @param string $name Nom du champ.
     * @param string $default Valeur par défaut (facultatif).
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $placeholder Clé de traduction pour le placeholder (facultatif).
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @return string HTML du champ de mot de passe.
     */
    public static function password(string $name, string $default = '', string $class = '', string $placeholder = '', string $desc = ''): string
    {
        global $lang;
        $value = Util::isPOST($name) ? self::clean($_POST[$name]) : $default; // Utilise la valeur POST ou par défaut
        $attrs = self::getAttributes($class, $placeholder, false, 'password'); // Génère les attributs HTML
        $descHtml = self::getDescription($desc); // Génère la description

        return sprintf(
            '<div class="form-group pass_show"><label for="%s">%s %s</label><input name="%s" value="%s" %s required autofocus>%s</div>',
            htmlspecialchars($name),
            $lang[$name] ?? $name, // Utilise la traduction ou le nom brut
            self::err($name . 'ErrNotMatch', $lang['errNotMatch'] ?? '') . self::err('bad_user_syntax', $lang['bad_user_syntax'] ?? ''), // Affiche les erreurs associées
            htmlspecialchars($name),
            htmlspecialchars($value),
            $attrs,
            $descHtml
        );
    }

    /**
     * Génère un champ de texte.
     *
     * @param string $name Nom du champ.
     * @param string $default Valeur par défaut (facultatif).
     * @param string $type Type du champ (par défaut : 'text').
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $placeholder Clé de traduction pour le placeholder (facultatif).
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @param bool $disabled Indique si le champ est désactivé (par défaut : false).
     * @return string HTML du champ de texte.
     */
    public static function text(string $name, string $default = '', string $type = 'text', string $class = '', string $placeholder = '', string $desc = '', bool $disabled = false): string
    {
        global $lang;
        $value = Util::isPOST($name) ? self::clean($_POST[$name]) : $default; // Utilise la valeur POST ou par défaut
        $attrs = self::getAttributes($class, $placeholder, $disabled, $type); // Génère les attributs HTML
        $descHtml = self::getDescription($desc); // Génère la description
        $errors = implode('', [
            self::err($name . 'ErrLen', $lang['errLen'] ?? ''),
            self::err($name . 'ErrNb', $lang['errNb'] ?? ''),
            self::err($name . 'the_specified_color_is_not_valid', $lang['the_specified_color_is_not_valid'] ?? ''),
            self::err($name . 'ErrContentFilter', $lang['ErrContentFilter'] ?? '')
        ]); // Combine toutes les erreurs possibles

        return sprintf(
            '<div class="form-group"><label for="%s">%s %s</label><input id="%s" name="%s" value="%s" %s>%s</div>',
            htmlspecialchars($name),
            $lang[$name] ?? $name,
            $errors,
            htmlspecialchars($name),
            htmlspecialchars($name),
            htmlspecialchars($value),
            $attrs,
            $descHtml
        );
    }

    /**
     * Génère un champ textarea.
     *
     * @param string $name Nom du champ.
     * @param string $default Valeur par défaut (facultatif).
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @param int $rows Nombre de lignes (par défaut : 10).
     * @param string $placeholder Clé de traduction pour le placeholder (facultatif).
     * @param bool $disabled Indique si le champ est désactivé (par défaut : false).
     * @return string HTML du champ textarea.
     */
    public static function textarea(string $name, string $default = '', string $class = '', string $desc = '', int $rows = 10, string $placeholder = '', bool $disabled = false): string
    {
        global $lang;
        $value = Util::isPOST($name) ? self::transNL(self::clean($_POST[$name])) : $default; // Nettoie et normalise la valeur POST ou par défaut
        $attrs = self::getAttributes($class, $placeholder, $disabled); // Génère les attributs HTML
        $descHtml = self::getDescription($desc); // Génère la description
        $errors = self::err($name . 'ErrLen', $lang['errLen'] ?? '') . self::err($name . 'ErrContentFilter', $lang['ErrContentFilter'] ?? ''); // Combine les erreurs possibles

        return sprintf(
            '<div class="form-group"><label for="%s">%s</label>%s<textarea id="%s" name="%s" rows="%d" %s>%s</textarea>%s%s</div>',
            htmlspecialchars($name),
            $lang[$name] ?? $name,
            Plugin::hook('editor'), // Ajoute un hook pour les éditeurs tiers
            htmlspecialchars($name),
            htmlspecialchars($name),
            $rows,
            $attrs,
            htmlspecialchars($value),
            $descHtml,
            $errors
        );
    }

    /**
     * Génère un bouton de soumission avec captcha.
     *
     * @param string $button Clé de traduction pour le texte du bouton (par défaut : 'submit').
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $icon Classe de l'icône à afficher (facultatif).
     * @param bool $cancel Indique si un bouton d'annulation doit être ajouté (par défaut : false).
     * @return string HTML du bouton de soumission et du captcha.
     */
    public static function submit(string $button = 'submit', string $class = '', string $icon = '', bool $cancel = false): string
    {
        global $lang;
        $classAttr = $class ? 'class="' . htmlspecialchars($class) . '"' : 'class="btn btn-primary btn-lg"'; // Définit la classe CSS
        $iconHtml = $icon ? '<i class="' . htmlspecialchars($icon) . '"></i> ' : ''; // Ajoute une icône si spécifiée
        $cancelButton = $cancel ? ' <button type="reset" class="btn btn-secondary btn-lg" onclick="$(\'#form\').remove();"><i class="fa fa-times" aria-hidden="true"></i> ' . htmlspecialchars($lang['cancel']) . '</button>' : ''; // Ajoute un bouton d'annulation si requis
        $captchaHtml = self::generateCaptcha(); // Génère le HTML du captcha

        return $captchaHtml . self::err('ErrToken', $lang['invalid_token'] ?? '') . self::err('ErrBot', $lang['errBot'] ?? '') .
               "<button $classAttr type=\"submit\">$iconHtml" . htmlspecialchars($lang[$button]) . "</button>$cancelButton"; // Combine tous les éléments
    }

    /**
     * Génère un bouton de soumission simple sans captcha.
     *
     * @param string $button Clé de traduction pour le texte du bouton (par défaut : 'submit').
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $icon Classe de l'icône à afficher (facultatif).
     * @param bool $cancel Indique si un bouton d'annulation doit être ajouté (par défaut : false).
     * @return string HTML du bouton de soumission.
     */
    public static function simple_submit(string $button = 'submit', string $class = '', string $icon = '', bool $cancel = false): string
    {
        global $lang;
        $classAttr = $class ? 'class="' . htmlspecialchars($class) . '"' : 'class="btn btn-primary"'; // Définit la classe CSS
        $iconHtml = $icon ? '<i class="' . htmlspecialchars($icon) . '"></i> ' : ''; // Ajoute une icône si spécifiée
        $cancelButton = $cancel ? ' <button type="reset" class="btn btn-secondary" onclick="$(\'#form\').remove();"><i class="fa fa-times" aria-hidden="true"></i> ' . htmlspecialchars($lang['cancel']) . '</button>' : ''; // Ajoute un bouton d'annulation si requis

        return self::err('ErrToken', $lang['invalid_token'] ?? '') . "<button $classAttr type=\"submit\">$iconHtml" . htmlspecialchars($lang[$button]) . "</button>$cancelButton"; // Combine les éléments
    }

    /**
     * Génère un menu déroulant (select).
     *
     * @param string $name Nom du champ.
     * @param array $options Options du menu (valeur => texte).
     * @param string $default Valeur par défaut (facultatif).
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @param bool $disabled Indique si le champ est désactivé (par défaut : false).
     * @return string HTML du menu déroulant.
     */
    public static function select(string $name, array $options, string $default = '', string $class = '', string $desc = '', bool $disabled = false): string
    {
        global $lang;
        $classAttr = $class ? 'class="form-control ' . htmlspecialchars($class) . '"' : 'class="custom-select"'; // Définit la classe CSS
        $descHtml = self::getDescription($desc); // Génère la description
        $disabledAttr = $disabled ? ' disabled' : ''; // Ajoute l'attribut disabled si nécessaire
        $selected = Util::isPOST($name) && isset($options[$_POST[$name]]) ? $_POST[$name] : $default; // Sélectionne la valeur POST ou par défaut

        $optionsHtml = '';
        foreach ($options as $value => $option) {
            $isSelected = $value == $selected ? ' selected="selected"' : ''; // Marque l'option comme sélectionnée si elle correspond
            $optionsHtml .= "<option value=\"" . htmlspecialchars($value) . "\"$isSelected>" . htmlspecialchars($option) . "</option>";
        }

        return sprintf(
            '<div class="form-group"><label class="form-label" for="%s">%s</label><select id="%s" name="%s" %s%s>%s</select>%s</div>',
            htmlspecialchars($name),
            $lang[$name] ?? $name, // Utilise la traduction ou le nom brut
            htmlspecialchars($name),
            htmlspecialchars($name),
            $classAttr,
            $disabledAttr,
            $optionsHtml,
            $descHtml
        );
    }

    /**
     * Génère une case à cocher.
     *
     * @param string $name Nom du champ.
     * @param string $default Valeur par défaut (facultatif).
     * @param string $desc Clé de traduction pour la description (facultatif).
     * @return string HTML de la case à cocher.
     */
    public static function checkBox(string $name, string $default = '', string $desc = ''): string
    {
        global $lang;
        $value = Util::isPOST($name) ? self::clean($_POST[$name]) : $default; // Utilise la valeur POST ou par défaut
        $descHtml = self::getDescription($desc); // Génère la description

        return sprintf(
            '<div class="form-group"><div class="custom-control custom-switch"><input class="custom-control-input" id="%s" name="%s" type="checkbox"%s><label class="custom-control-label" for="%s">%s</label></div>%s</div>',
            htmlspecialchars($name),
            htmlspecialchars($name),
            $value ? ' checked' : '', // Coche la case si la valeur existe
            htmlspecialchars($name),
            $lang[$name] ?? $name, // Utilise la traduction ou le nom brut
            $descHtml
        );
    }

    /**
     * Génère un formulaire HTML.
     *
     * @param string $action URL de l'action du formulaire.
     * @param string $controls Contenu HTML des champs du formulaire.
     * @param string $class Classes CSS supplémentaires (facultatif).
     * @param string $method Méthode HTTP (par défaut : 'post').
     * @param bool $enctype Indique si l'enctype multipart/form-data est utilisé (par défaut : true).
     * @return string HTML du formulaire.
     */
    public static function form(string $action, string $controls, string $class = '', string $method = 'post', bool $enctype = true): string
    {
        global $token;
        $classAttr = $class ? 'class="' . htmlspecialchars($class) . '"' : ''; // Définit la classe CSS
        $enctypeAttr = $enctype ? ' enctype="multipart/form-data"' : ''; // Ajoute l'enctype si nécessaire

        return sprintf(
            '<form id="form" action="%s" method="%s" %s%s><input type="hidden" name="_token" value="%s">%s</form>',
            htmlspecialchars($action),
            htmlspecialchars($method),
            $classAttr,
            $enctypeAttr,
            htmlspecialchars($token), // Ajoute un jeton CSRF
            $controls
        );
    }

    /**
     * Génère un slug pour les fichiers téléchargés.
     *
     * @param string $text Texte à convertir en slug.
     * @return string Slug nettoyé et normalisé.
     */
    public static function slugupload(string $text): string
    {
        $find = ["/Ğ/","/Ü/","/Ş/","/İ/","/Ö/","/Ç/","/ğ/","/ü/","/ş/","/ı/","/ö/","/ç/", "/I/"];
        $replace = ["g","u","s","i","o","c","g","u","s","i","o","c","i"];
        $text = preg_replace("/[^0-9a-zA-ZÄzÜŞİÖÇğüşıöç]/", " ", $text); // Remplace les caractères non alphanumériques par des espaces
        $text = preg_replace($find, $replace, $text); // Remplace les caractères spéciaux turcs
        $text = preg_replace("/ +/", " ", $text); // Réduit les espaces multiples
        $text = preg_replace("/ /", "-", $text); // Remplace les espaces par des tirets
        $text = preg_replace("/\s/", "", $text); // Supprime les espaces restants
        $text = strtolower($text); // Convertit en minuscules
        return trim($text, '-'); // Supprime les tirets aux extrémités
    }

    /**
     * Convertit une taille en octets.
     *
     * @param string $size Taille avec unité (ex. '10MB').
     * @return float Taille en octets.
     */
    private static function convertToBytes(string $size): float
    {
        $size = floatval($size); // Convertit en nombre
        if (strpos($size, 'KB')) {
            return $size * 1024; // Convertit les KB en octets
        }
        if (strpos($size, 'MB')) {
            return $size * 1048576; // Convertit les MB en octets
        }
        if (strpos($size, 'GB')) {
            return $size * 1073741824; // Convertit les GB en octets
        }
        return $size; // Retourne la taille brute si aucune unité
    }

    /**
     * Gère le téléchargement de fichiers.
     *
     * @param string $name Nom du champ de fichier.
     * @param string $dir Répertoire de destination.
     * @param string $size Taille maximale autorisée.
     * @return array Résultat du téléchargement (status, message, nom du fichier).
     */
    public static function upload(string $name, string $dir, string $size): array
    {
        global $lang;
        if (!isset($_FILES[$name]) || empty($_FILES[$name]["name"])) {
            return ["status" => "error", "msg" => $lang['please_select_file'] ?? "Veuillez sélectionner un fichier.", "name" => ""]; // Erreur si aucun fichier
        }

        $maxSize = self::convertToBytes($size); // Convertit la taille max en octets
        $fileInfo = pathinfo($_FILES[$name]["name"]); // Extrait les informations du fichier
        $filename = self::slugupload($fileInfo["filename"]) . '.' . $fileInfo["extension"]; // Génère un nom de fichier sécurisé
        $uniqueFilename = rand(0, 999) . uniqid() . $filename; // Ajoute un préfixe unique
        $filePath = rtrim($dir, '/') . '/' . $uniqueFilename; // Construit le chemin complet

        if ($_FILES[$name]["size"] > $maxSize) {
            return ["status" => "error", "msg" => $lang['file_size_exceeded'] ?? "La taille du fichier dépasse la limite maximale !", "name" => ""]; // Erreur si taille dépassée
        }

        if (move_uploaded_file($_FILES[$name]["tmp_name"], $filePath)) {
            return ["status" => "success", "msg" => $lang['file_uploaded'] ?? "Le fichier a été téléchargé avec succès !", "name" => $filePath]; // Succès
        }

        return ["status" => "error", "msg" => $lang['upload_error'] ?? "Une erreur s'est produite lors du téléchargement !", "name" => ""]; // Erreur générique
    }

    /**
     * Génère un aperçu du contenu.
     *
     * @param string $name Nom du champ.
     * @return string HTML de l'aperçu ou chaîne vide.
     */
    public static function preview(string $name): string
    {
        return Util::isPOST($name) ? '<div class="alert alert-warning" role="alert">' . Parser::content(self::transNL(self::clean($_POST[$name]))) . '</div>' : ''; // Affiche l'aperçu si valeur POST
    }

    /**
     * Valide un champ de texte.
     *
     * @param string $name Nom du champ.
     * @param int $min Longueur minimale (par défaut : 1).
     * @param int $max Longueur maximale (par défaut : 40).
     * @return bool True si valide, false sinon.
     */
    public static function check(string $name, int $min = 1, int $max = 40): bool
    {
        if (!Util::isPOST($name) || !isset($_POST[$name])) {
            return false; // Invalide si pas de valeur POST
        }

        $content = filter_var($_POST[$name], FILTER_SANITIZE_FULL_SPECIAL_CHARS); // Nettoie le contenu
        if (preg_match(CONTENT_FILTER, $content)) {
            $_SESSION[$name . 'ErrContentFilter'] = 1; // Erreur si contenu interdit
            return false;
        }

        $len = strlen(trim($content)); // Calcule la longueur
        if ($len < $min || $len > $max) {
            $_SESSION[$name . 'ErrLen'] = 1; // Erreur si longueur hors limites
            return false;
        }

        return true; // Valide
    }

    /**
     * Valide la correspondance de deux mots de passe.
     *
     * @param string $name Nom du champ de mot de passe.
     * @return bool True si les mots de passe correspondent, false sinon.
     */
    public static function checkPass(string $name): bool
    {
        if (self::check($name) && Util::isPOST($name . 'Confirm') && $_POST[$name] === $_POST[$name . 'Confirm']) {
            return true; // Valide si le mot de passe et sa confirmation correspondent
        }
        $_SESSION[$name . 'ErrNotMatch'] = 1; // Erreur si non correspondance
        return false;
    }

    /**
     * Valide un champ numérique.
     *
     * @param string $name Nom du champ.
     * @return bool True si valide, false sinon.
     */
    public static function checkNb(string $name): bool
    {
        if (isset($_POST[$name]) && ctype_digit($_POST[$name]) && $_POST[$name] > 0) {
            return true; // Valide si c'est un nombre positif
        }
        $_SESSION[$name . 'ErrNb'] = 1; // Erreur si non numérique ou non positif
        return false;
    }

    /**
     * Valide une adresse e-mail.
     *
     * @param string $email Adresse e-mail à valider.
     * @return bool True si valide, false sinon.
     */
    public static function checkMail(string $email): bool
    {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false; // Valide le format de l'e-mail
    }

    /**
     * Valide un captcha.
     *
     * @return bool True si le captcha est correct, false sinon.
     */
    public static function checkBot(): bool
    {
        if (!Util::isPOST('captcha') || !isset($_SESSION['captcha']) || $_POST['captcha'] !== $_SESSION['captcha']) {
            $_SESSION['ErrBot'] = 1; // Erreur si captcha incorrect
            return false;
        }
        return true; // Valide
    }

    /**
     * Vérifie si un captcha est requis en fonction de la session.
     *
     * @return bool True si aucun captcha n'est requis, false sinon.
     */
    public static function sessionTrip(): bool
    {
        global $sessionTrip;
        return $sessionTrip || User::isWorker() || self::checkBot(); // Vérifie les conditions pour bypasser le captcha
    }

    /**
     * Génère un bouton de soumission en fonction de la session.
     *
     * @param string $name Clé de traduction pour le texte du bouton (par défaut : 'submit').
     * @param string $icon Classe de l'icône (par défaut : 'fa fa-paper-plane').
     * @param string $default Valeur par défaut (facultatif).
     * @return string HTML du bouton.
     */
    public static function tripCaptcha(string $name = 'submit', string $icon = 'fa fa-paper-plane', string $default = ''): string
    {
        return self::sessionTrip() ? self::simple_submit($name, $default, $icon) : self::submit($name, $default, $icon); // Choisit entre simple_submit et submit selon session
    }

    /**
     * Génère le HTML pour un captcha.
     *
     * @return string HTML du captcha.
     */
    private static function generateCaptcha(): string
    {
        global $lang;
        if (!CAPTCHA) {
            return sprintf(
                '<div class="input-group mb-3"><div class="input-group-prepend"><span class="input-group-text">%s<img id="cap-img" src="%s/lib/Captcha.lib.php?rand=%s" alt="%s"/></span><a class="input-group-text" onClick="reload();" data-toggle="tooltip" data-placement="top" title="%s"><i class="fa fa-refresh" aria-hidden="true"></i></a></div><label for="captcha" class="sr-only">%s</label><input name="captcha" type="text" class="form-control" placeholder="%s"></div><script>function reload() { var img = document.images["cap-img"]; img.src = img.src.substring(0, img.src.lastIndexOf("?")) + "?rand=" + Math.random() * 1000; }</script>',
                Plugin::hook('captcha'), // Ajoute un hook pour les captchas tiers
                HTML_BASEPATH . DS,
                rand(), // Génère un nombre aléatoire pour l'image
                htmlspecialchars($lang['captcha'] ?? 'Captcha'),
                htmlspecialchars($lang['r_captcha'] ?? 'Recharger Captcha'),
                htmlspecialchars($lang['captcha'] ?? 'Captcha'),
                htmlspecialchars($lang['enter_code'] ?? 'Entrez le code')
            ); // Génère un captcha basé sur une image
        }

        $num1 = mt_rand(1, 20);
        $num2 = mt_rand(1, 20);
        $isAddition = mt_rand(0, 1) === 1;
        $math = $isAddition ? "$num1 + $num2" : "$num1 - $num2"; // Génère une opération mathématique simple
        $_SESSION['captcha'] = (string)($isAddition ? $num1 + $num2 : $num1 - $num2); // Stocke la réponse en session

        return sprintf(
            '<div class="input-group mb-3"><div class="input-group-prepend"><span class="input-group-text">%s = ?</span></div><label for="captcha" class="sr-only">%s</label><input name="captcha" type="text" class="form-control" placeholder="%s"></div>',
            $math,
            htmlspecialchars($lang['captcha'] ?? 'Captcha'),
            htmlspecialchars($lang['math_result'] ?? 'Entrez le résultat')
        ); // Génère un captcha mathématique
    }
}