  _  _     _   _ _/   _
# / (   (- (/ (/ (- _)  /  _)
#          /

"""
Requests HTTP Library
~~~~~~~~~~~~~~~~~~~~~

Requests is an HTTP library, written in Python, for human beings. Basic GET
usage:

   >>> import requests
   >>> r = requests.get('https://www.python.org')
   >>> r.status_code
   200
   >>> 'Python is a programming language' in r.content
   True

... or POST:

   >>> payload = dict(key1='value1', key2='value2')
   >>> r = requests.post('https://httpbin.org/post', data=payload)
   >>> print(r.text)
   {
     ...
     "form": {
       "key2": "value2",
       "key1": "value1"
     },
     ...
   }

The other HTTP methods are supported - see `requests.api`. Full documentation
is at <http://python-requests.org>.

:copyright: (c) 2017 by Kenneth Reitz.
:license: Apache 2.0, see LICENSE for more details.
"""

from pip._vendor import urllib3
from pip._vendor import chardet
import warnings
from .exceptions import RequestsDependencyWarning


def check_compatibility(urllib3_version, chardet_version):
    urllib3_version = urllib3_version.split('.')
    assert urllib3_version != ['dev']  # Verify urllib3 isn't installed from git.

    # Sometimes, urllib3 only reports its version as 16.1.
    if len(urllib3_version) == 2:
        urllib3_version.append('0')

    # Check urllib3 for compatibility.
    major, minor, patch = urllib3_version  # noqa: F811
    major, minor, patch = int(major), int(minor), int(patch)
    # urllib3 >= 1.21.1, <= 1.25
    assert major == 1
    assert minor >= 21
    assert minor <= 25

    # Check chardet for compatibility.
    major, minor, patch = chardet_version.split('.')[:3]
    major, minor, patch = int(major), int(minor), int(patch)
    # chardet >= 3.0.2, < 3.1.0
    assert major == 3
    assert minor < 1
    assert patch >= 2


def _check_cryptography(cryptography_version):
    # cryptography < 1.3.4
    try:
        cryptography_version = list(map(int, cryptography_version.split('.')))
    except ValueError:
        return

    if cryptography_version < [1, 3, 4]:
        warning = 'Old version of cryptography ({}) may cause slowdown.'.format(cryptography_version)
        warnings.warn(warning, RequestsDependencyWarning)

# Check imported dependencies for compatibility.
try:
    check_compatibility(urllib3.__version__, chardet.__version__)
except (AssertionError, ValueError):
    warnings.warn("urllib3 ({}) or chardet ({}) doesn't match a supported "
                  "version!".format(urllib3.__version__, chardet.__version__),
                  RequestsDependencyWarning)

# Attempt to enable urllib3's SNI support, if possible
from pip._internal.utils.compat import WINDOWS
if not WINDOWS:
    try:
        from pip._vendor.urllib3.contrib import pyopenssl
        pyopenssl.inject_into_urllib3()

        # Check cryptography version
        from cryptography import __version__ as cryptography_version
        _check_cryptography(cryptography_version)
    except ImportError:
        pass

# urllib3's DependencyWarnings should be silenced.
from pip._vendor.urllib3.exceptions import DependencyWarning
warnings.simplefilter('ignore', DependencyWarning)

from .__version__ import __title__, __description__, __url__, __version__
from .__version__ import __build__, __author__, __author_email__, __license__
from .__version__ import __copyright__, __cake__

from . import utils
from . import packages
from .models import Request, Response, PreparedRequest
from .api import request, get, head, post, patch, put, delete, options
from .sessions import session, Session
from .status_codes import codes
from .exceptions import (
    RequestException, Timeout, URLRequired,
    TooManyRedirects, HTTPError, ConnectionError,
    FileModeWarning, ConnectTimeout, ReadTimeout
)

# Set default logging handler to avoid "No handler found" warnings.
import logging
from logging import NullHandler

logging.getLogger(__name__).addHandler(NullHandler())

# FileModeWarnings go off per the default.
warnings.simplefilter('default', FileModeWarning, append=True)
PK       !       requests/certs.pynu [        #!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
requests.certs
~~~~~~~~~~~~~~

This module returns the preferred default CA certificate bundle. There is
only one — the one from the certifi package.

If you are packaging Requests, e.g., for a Linux distribution or a managed
environment, you can change the definition of where() to return a separately
packaged CA bundle.
"""
from pip._vendor.certifi import where

if __name__ == '__main__':
    print(where())
PK       ! 1E    *  __pycache__/ipaddress.cpython-38.opt-1.pycnu [        U

    .e7                    @   s`  d Z ddlmZ ddlZddlZdZefZzeefZW n e	k
rJ   Y nX ze
ZW n e	k
rl   eZY nX ddkrdd Z
ndd Z
z
ejZW n ek
r   dd	 ZY nX d
d Zeedrd
d Zndd ZdNddZG dd deZdZdZG dd deZG dd deZdd ZdOddZd d! Zd"d# Zd$d% Z d&d' Z!d(d) Z"d*d+ Z#d,d- Z$d.d/ Z%d0d1 Z&d2d3 Z'G d4d5 d5eZ(G d6d7 d7e(Z)G d8d9 d9e(Z*G d:d; d;eZ+G d<d= d=e+e)Z,G d>d? d?e,Z-G d@dA dAe+e*Z.G dBdC dCeZ/e/e,_0G dDdE dEeZ1G dFdG dGe1e)Z2G dHdI dIe2Z3G dJdK dKe1e*Z4G dLdM dMeZ5e5e2_0dS )PzA fast, lightweight IPv4/IPv6 manipulation library in Python.

This library is used to create/poke/manipulate IPv4 and IPv6 addresses
and networks.

    )unicode_literalsNz1.0.22c                 C   s   | S N Zbytr   r   9/usr/lib/python3.8/site-packages/pip/_vendor/ipaddress.py_compat_bytes_to_byte_vals   s    r   c                 C   s   dd | D S )Nc                 S   s   g | ]}t d |d qS )   !Br   )structunpack.0br   r   r   
<listcomp>#   s     z._compat_bytes_to_byte_vals.<locals>.<listcomp>r   r   r   r   r   r   "   s    c                 C   s   d}| D ]}|d> | }q|S )Nr      r   )Zbytvals	endianessresZbvr   r   r   _compat_int_from_byte_vals'   s    r   c                 C   st   |dkr.| dk s| dkr"t dt d| S |dkrj| dk sJ| dd krTt d	t d
| d? | d@ S t d S )
N   r   l        z(integer out of range for 'I' format codes   !I         z)integer out of range for 'QQ' format codes   !QQ@   l    )r	   errorpackNotImplementedError)ZintvalZlengthr   r   r   r   _compat_to_bytes0   s    

r   
bit_lengthc                 C   s   |   S r   )r   )ir   r   r   _compat_bit_lengthA   s    r   c                 C   s&   t  D ]}| |? dkr|  S qd S Nr   )	itertoolscount)r   r   r   r   r   r   D   s       c                 c   s    | }||k r|V  ||7 }qd S r   r   )startendstepr   r   r   r   
_compat_rangeJ   s    r&   c                   @   s@   e Zd ZdZdd Zdd Zdd Zdd	 Zd
d Zdd
 Z	dS )_TotalOrderingMixinr   c                 C   s   t d S r   r   selfotherr   r   r   __eq__Y   s    z_TotalOrderingMixin.__eq__c                 C   s   |  |}|tkrtS | S r   )r,   NotImplemented)r*   r+   equalr   r   r   __ne__\   s    
z_TotalOrderingMixin.__ne__c                 C   s   t d S r   r(   r)   r   r   r   __lt__b   s    z_TotalOrderingMixin.__lt__c                 C   s$   |  |}|tks|s | |S |S r   r0   r-   r,   r*   r+   lessr   r   r   __le__e   s    

z_TotalOrderingMixin.__le__c                 C   s6   |  |}|tkrtS | |}|tkr,tS |p2| S r   r1   )r*   r+   r3   r.   r   r   r   __gt__k   s    

z_TotalOrderingMixin.__gt__c                 C   s   |  |}|tkrtS | S r   )r0   r-   r2   r   r   r   __ge__t   s    
z_TotalOrderingMixin.__ge__N)
__name__
__module____qualname__	__slots__r,   r/   r0   r4   r5   r6   r   r   r   r   r'   R   s   	r'       r   c                   @   s   e Zd ZdZdS )AddressValueErrorz%A Value Error related to the address.Nr7   r8   r9   __doc__r   r   r   r   r<      s   r<   c                   @   s   e Zd ZdZdS )NetmaskValueErrorz%A Value Error related to the netmask.Nr=   r   r   r   r   r?      s   r?   c              	   C   sn   z
t | W S  ttfk
r"   Y nX z
t| W S  ttfk
rF   Y nX t| tr^td|  td|  dS )a  Take an IP string/int and return an object of the correct type.

    Args:
        address: A string or integer, the IP address.  Either IPv4 or
          IPv6 addresses may be supplied; integers less than 2**32 will
          be considered to be IPv4 by default.

    Returns:
        An IPv4Address or IPv6Address object.

    Raises:
        ValueError: if the *address* passed isn't either a v4 or a v6
          address

    zx%r does not appear to be an IPv4 or IPv6 address. Did you pass in a bytes (str in Python 2) instead of a unicode object?z0%r does not appear to be an IPv4 or IPv6 addressN)IPv4Addressr<   r?   IPv6Address
isinstancebytes
ValueErroraddressr   r   r   
ip_address   s"    


rG   Tc              	   C   sr   zt | |W S  ttfk
r$   Y nX zt| |W S  ttfk
rJ   Y nX t| trbtd|  td|  dS )a  Take an IP string/int and return an object of the correct type.

    Args:
        address: A string or integer, the IP network.  Either IPv4 or
          IPv6 networks may be supplied; integers less than 2**32 will
          be considered to be IPv4 by default.

    Returns:
        An IPv4Network or IPv6Network object.

    Raises:
        ValueError: if the string passed isn't either a v4 or a v6
          address. Or if the network has host bits set.

    zx%r does not appear to be an IPv4 or IPv6 network. Did you pass in a bytes (str in Python 2) instead of a unicode object?z0%r does not appear to be an IPv4 or IPv6 networkN)IPv4Networkr<   r?   IPv6NetworkrB   rC   rD   )rF   strictr   r   r   
ip_network   s"    
rK   c              	   C   sX   z
t | W S  ttfk
r"   Y nX z
t| W S  ttfk
rF   Y nX td|  dS )ag  Take an IP string/int and return an object of the correct type.

    Args:
        address: A string or integer, the IP address.  Either IPv4 or
          IPv6 addresses may be supplied; integers less than 2**32 will
          be considered to be IPv4 by default.

    Returns:
        An IPv4Interface or IPv6Interface object.

    Raises:
        ValueError: if the string passed isn't either a v4 or a v6
          address.

    Notes:
        The IPv?Interface classes describe an Address on a particular
        Network, so they're basically a combination of both the Address
        and Network classes.

    z2%r does not appear to be an IPv4 or IPv6 interfaceN)
IPv4Interfacer<   r?   
IPv6InterfacerD   rE   r   r   r   ip_interface   s    

rN   c              	   C   s6   zt | ddW S  tjtfk
r0   tdY nX dS )a`  Represent an address as 4 packed bytes in network (big-endian) order.

    Args:
        address: An integer representation of an IPv4 IP address.

    Returns:
        The integer address packed as 4 bytes in network (big-endian) order.

    Raises:
        ValueError: If the integer is negative or too large to be an
          IPv4 IP address.

    r   bigz&Address negative or too large for IPv4Nr   r	   r   
OverflowErrorrD   rE   r   r   r   v4_int_to_packed   s    rR   c              	   C   s6   zt | ddW S  tjtfk
r0   tdY nX dS )zRepresent an address as 16 packed bytes in network (big-endian) order.

    Args:
        address: An integer representation of an IPv6 IP address.

    Returns:
        The integer address packed as 16 bytes in network (big-endian) order.

    r   rO   z&Address negative or too large for IPv6NrP   rE   r   r   r   v6_int_to_packed  s    
rS   c                 C   s*   t | d}t|dkr&td|  |S )zAHelper to split the netmask and raise AddressValueError if needed/r   zOnly one '/' permitted in %r)_compat_strsplitlenr<   )rF   addrr   r   r   _split_optional_netmask  s    rY   c                 c   sN   t | }t| }}|D ]&}|j|jd kr:||fV  |}|}q||fV  dS )zFind a sequence of sorted deduplicated IPv#Address.

    Args:
        addresses: a list of IPv#Address objects.

    Yields:
        A tuple containing the first and last IP addresses in the sequence.

    r"   N)iternext_ip)	addressesitfirstlastipr   r   r   _find_address_range  s    

rb   c                 C   s$   | dkr|S t |t|  | d @ S )zCount the number of zero bits on the right hand side.

    Args:
        number: an integer.
        bits: maximum number of bits to count.

    Returns:
        The number of zero bits on the right hand side of the number.

    r   r"   )minr   )Znumberbitsr   r   r   _count_righthand_zero_bits2  s    re   c                 c   s   t | trt |tstd| j|jkr8td| |f | |krHtd| jdkrXt}n| jdkrht}ntd| j}| j}|j}||krt	t
||t|| d d }|||| f}|V  |d|> 7 }|d |jkrqqdS )	a  Summarize a network range given the first and last IP addresses.

    Example:
        >>> list(summarize_address_range(IPv4Address('192.0.2.0'),
        ...                              IPv4Address('192.0.2.130')))
        ...                                #doctest: +NORMALIZE_WHITESPACE
        [IPv4Network('192.0.2.0/25'), IPv4Network('192.0.2.128/31'),
         IPv4Network('192.0.2.130/32')]

    Args:
        first: the first IPv4Address or IPv6Address in the range.
        last: the last IPv4Address or IPv6Address in the range.

    Returns:
        An iterator of the summarized IPv(4|6) network objects.

    Raise:
        TypeError:
            If the first and last objects are not IP addresses.
            If the first and last objects are not the same version.
        ValueError:
            If the last object is not greater than the first.
            If the version of the first address is not 4 or 6.

    z1first and last must be IP addresses, not networks%%s and %s are not of the same versionz*last IP address must be greater than firstr      zunknown IP versionr"   N)
rB   _BaseAddress	TypeErrorversionrD   rH   rI   _max_prefixlenr\   rc   re   r   	_ALL_ONES)r_   r`   ra   Zip_bitsZ	first_intZlast_intZnbitsnetr   r   r   summarize_address_rangeB  s8    
 


rn   c                 c   s   t | }i }|rV| }| }||}|dkr<|||< q||kr||= || qd}t| D ]$}|dk	r|j|jkrqf|V  |}qfdS )au  Loops through the addresses, collapsing concurrent netblocks.

    Example:

        ip1 = IPv4Network('192.0.2.0/26')
        ip2 = IPv4Network('192.0.2.64/26')
        ip3 = IPv4Network('192.0.2.128/26')
        ip4 = IPv4Network('192.0.2.192/26')

        _collapse_addresses_internal([ip1, ip2, ip3, ip4]) ->
          [IPv4Network('192.0.2.0/24')]

        This shouldn't be called directly; it is called via
          collapse_addresses([]).

    Args:
        addresses: A list of IPv4Network's or IPv6Network's

    Returns:
        A list of IPv4Network's or IPv6Network's depending on what we were
        passed.

    N)listpopsupernetgetappendsortedvaluesbroadcast_address)r]   Zto_mergesubnetsrm   rq   Zexistingr`   r   r   r   _collapse_addresses_internaly  s$    

rx   c              	   C   s0  g }g }g }| D ]}t |trR|rF|d j|jkrFtd||d f || q|j|jkr|r|d j|jkrtd||d f z||j W q tk
r   ||j	 Y qX q|r|d j|jkrtd||d f || qt
t|}|r$t|D ]\}}|
t|| qt|| S )a  Collapse a list of IP objects.

    Example:
        collapse_addresses([IPv4Network('192.0.2.0/25'),
                            IPv4Network('192.0.2.128/25')]) ->
                           [IPv4Network('192.0.2.0/24')]

    Args:
        addresses: An iterator of IPv4Network or IPv6Network objects.

    Returns:
        An iterator of the collapsed IPv(4|6)Network objects.

    Raises:
        TypeError: If passed a list of mixed version objects.

    rf   )rB   rh   _versionri   rs   
_prefixlenrk   ra   AttributeErrornetwork_addressrt   setrb   extendrn   rx   )r]   ZaddrsZipsZnetsra   r_   r`   r   r   r   collapse_addresses  s@    
   r   c                 C   s(   t | tr|  S t | tr$|  S tS )a2  Return a key suitable for sorting between networks and addresses.

    Address and Network objects are not sortable by default; they're
    fundamentally different so the expression

        IPv4Address('192.0.2.0') <= IPv4Network('192.0.2.0/24')

    doesn't make any sense.  There are some times however, where you may wish
    to have ipaddress sort these for you anyway. If you need to do this, you
    can use this function as the key= argument to sorted().

    Args:
      obj: either a Network or Address object.
    Returns:
      appropriate key.

    )rB   _BaseNetwork_get_networks_keyrh   _get_address_keyr-   )objr   r   r   get_mixed_type_key  s
    

r   c                   @   s   e Zd ZdZdZedd Zedd Zedd Zed	d
 Z	dd Z
d
d Zedd Z
edd Zedd Zedd Zedd Zdd ZdS )_IPAddressBasezThe mother class.r   c                 C   s   |   S )z:Return the longhand version of the IP address as a string.)_explode_shorthand_ip_stringr*   r   r   r   exploded  s    z_IPAddressBase.explodedc                 C   s   t | S )z;Return the shorthand version of the IP address as a string.rU   r   r   r   r   
compressed  s    z_IPAddressBase.compressedc                 C   s   |   S )aI  The name of the reverse DNS pointer for the IP address, e.g.:
            >>> ipaddress.ip_address("127.0.0.1").reverse_pointer
            '1.0.0.127.in-addr.arpa'
            >>> ipaddress.ip_address("2001:db8::1").reverse_pointer
            '1.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.8.b.d.0.1.0.0.2.ip6.arpa'

        )_reverse_pointerr   r   r   r   reverse_pointer	  s    	z_IPAddressBase.reverse_pointerc                 C   s   dt | f }t|d S )Nz%200s has no version specifiedtyper   r*   msgr   r   r   rj     s    z_IPAddressBase.versionc                 C   sF   |dk rd}t ||| jf || jkrBd}t ||| j| jf d S )Nr   z-%d (< 0) is not permitted as an IPv%d addressz2%d (>= 2**%d) is not permitted as an IPv%d address)r<   rz   rl   rk   )r*   rF   r   r   r   r   _check_int_address  s    

z!_IPAddressBase._check_int_addressc                 C   s.   t |}||kr*d}t||||| jf d S )Nz~%r (len %d != %d) is not permitted as an IPv%d address. Did you pass in a bytes (str in Python 2) instead of a unicode object?)rW   r<   rz   )r*   rF   Zexpected_lenZaddress_lenr   r   r   r   _check_packed_address"  s     z$_IPAddressBase._check_packed_addressc                 C   s   | j | j |? A S )zTurn the prefix length into a bitwise netmask

        Args:
            prefixlen: An integer, the prefix length.

        Returns:
            An integer.

        )rl   )cls	prefixlenr   r   r   _ip_int_from_prefix,  s    z"_IPAddressBase._ip_int_from_prefixc           	      C   s\   t || j}| j| }||? }d|> d }||krX| jd }t||d}d}t|| |S )a  Return prefix length from the bitwise netmask.

        Args:
            ip_int: An integer, the netmask in expanded bitwise format

        Returns:
            An integer, the prefix length.

        Raises:
            ValueError: If the input intermingles zeroes & ones
        r"   r   rO   z&Netmask pattern %r mixes zeroes & ones)re   rk   r   rD   )	r   ip_intZtrailing_zeroesr